/*
 * ug.space.c --Spatial Processing Unit Generator originally written by
 *		F. Richard Moore as part of the cmusic program in the 80s.
 *		Improvements and port to Pd and Max/MSP by Shahrokh Yadegari.
 *		- for bugs and suggestions send email to sdy@ucsd.edu
 *
 *		Dec. 2004 version 0.5 --sdy
 *		-Added the inner room model
 *
 *		Oct. 2004 version 0.4 --sdy
 *		-Added room managment, now every space object has its own room
 *		-Added room definition as arguments to the object
 *		 now the usage is as follows
 *		 space [card] [stereo|quad] [innersize] [outsersize]
 *		 "card" adds inputs for real time inputs of
 *		 "theta" and "back", for cardiod pattern defitions
 *		Alpha release
 *
 *		Feb 2002 version 0.3 --sdy
 *		-Fixed bugs in the interpolation scheme
 *		 the doppler shift is heard more prominently now
 *		-Fixed bugs for when back is less than 1 or when
 *		 theta was much larger than PI
 *
 *		April 2001 version 0.2 --sdy
 *		-Added interpolation code for efficiency
 *		-Added a new getcut() code by Harry Castle
 *
 * 		March 2000
 *		The current code assumes a single room and all the instances of
 *		space in REALTIME or cmusic use the same definition.
 *		A number of variables have to change to allow multiple
 *		room definitions for each different instances of space~ in pd
 *		which means moving the room definition variables in the
 *		t_space_tilde structure
 *
 */

/* to compile for MSP uncomment this following line*/
// #define MSP

#include <stdlib.h>
#include <stdio.h>
#include <stdio.h>

#if defined(PD)||defined(MSP)
#define REALTIME
#endif

#ifndef REALTIME
#include "mm.head.h"
#include "ug.head.h"

#define UNUSED	1
#define DBUF	2
#define DLEN	3
#define NOW	4
#define DARRAY	5
#define CARRAY	6
#define AARRAY	7
#define FARRAY	8
#define XS	9
#define YS	10
#define THETAS	11
#define AMPS	12
#define BACKS	13
#define X	14
#define Y	15
#define THETA   16
#define AMP     17
#define BACK    18

#else /* ifndef REALTIME */

#include "ug.defs.h"

#define	Srate	((long) x->sp_srate)	/* sampling rate */
#define	Nchan 	(x->sp_nchan) /* number of channels */
#define	Ngen	(x->sp_vsize) /* Samples to generate at 1 time (<= Lblocks) */
#define	Grevblock	(x->sp_grev) /* Pointer to global reverb output block */
#define Outblock	((float *) x->sp_out)

#ifdef notdef /* SDY */
/* moved to space~.h */

/* the following is out of cmusic:m.globs.h */
float Lx[]={5.,-5.,-5.,5.}; /* Listening space x-coordinates */
float Ly[]={5.,5.,-5.,-5.}; /* Listening space y-coordinates */
int   NLs = 4;		/* Number of listening surfaces */
float Ax[]={50.,-50.,-50.,50.};	/* Acoustic space x-coordinates */
float Ay[]={50.,50.,-50.,-50};	/* Acoustic space y-coordinates */
int   NAs = 4;		/* Number of acoustic space surfaces */
float Sx[]={5.,-5.,-5.,5.}; /* Speaker x-coordinates */
float Sy[]={5.,5.,-5.,-5.}; /* Speaker y-coordinates */
float Revscale=.08;	/* Global reverb input scale */
float T60mult=.83;	/* T60 multiplier */
float Revthresh=.0001;	/* Reverb tail cancellation threshold */
float Direct = 1.;	/* Direct path distance amplifier */
float Reflect = 1.;	/* Reflected path distance amplifier */
#endif

int Spaceon;		/* Spatial processing flag */
int Spacewason;		/* Spatial processing souvenir */
int Spacereset;		/* Spatial processing reset flag */
float Maxecho;		/* Maximum echo amplitude generated */

int space_debug;
int space_delinterpol;	/* what type of interpolation */

#endif

#include <stdio.h>
#include <math.h>


#define CI	.00029851	/* 1/335 (meters per second) */
#define PI2	6.283185308
#ifndef MSP
#define PI	3.141592654
#endif /* !MSP */
#define IPI	.318309886
#define D270	4.712388981
#define D90	1.570796327
#define DIST(x1,y1,x2,y2) sqrt( (double) ( (x1-x2)*(x1-x2) + (y1-y2)*(y1-y2) ) )
#define TAU	50		/* milliseconds in cross-fade */

float ***fp3alloc(int x, int y, int z); 
void fp3free(float ***fp3, int x, int y, int z);
float do_delay(float buf[],long len, long now, float tau);
float getcut(t_space_room *,
	float x1, float sp_y1, float x2, float y2, int fracret);
void getrefl(t_space_room *,
	float sx, float sy, float cx, float cy, int s,float *rx,float *ry);
int within(t_space_room *, float x,float y);
int space_mkroom(t_space_tilde *x);

//SDY delete 
float
ngetcut(t_space_room *rm, float x1, float y_1, float x2, float y2, int retfrac);

void
#ifndef REALTIME
space_init(narg, ap)
register long narg;
register union arg_ptr ap[];
#else
space_init(t_space_tilde *x)
#endif
{
	float maxdist, maxdelay, d;
	float maxspdist;	/* Max speaker dist */
	long dlen, now, j, k, nrv, c, r, s, ns;
	/*
	float RCI = CI*Srate;
	*/
	float RCI = CI * SP_MAX_SRATE;
	float NAs;
	t_space_room *rm;



	Spacewason = Spaceon = 1;	/* Turn on global reverberator */
	maxdist = 0.;
	rm = x->sp_room;	/* the first room is the biggest */
	NAs = rm->spr_NAs;
	for(j=0; j<NAs; j++){ /* Find largest distance in acoustic space */
	    for(k=0; k<NAs; k++){
		if(j == k) continue;
		d = DIST(rm->spr_Ax[j], rm->spr_Ay[j],
						rm->spr_Ax[k], rm->spr_Ay[k]);
		if(d > maxdist) maxdist = d;
	    }
	}
	/*
	 * find the largest speaker delay
	 */
	maxspdist = 0;
	for (j = 0; j < rm->spr_ns; j++) {
		if (rm->spr_Sdelay[j] > maxspdist)
			maxspdist = rm->spr_Sdelay[j];
	}
	maxdelay = 2.*maxdist + maxspdist;
	dlen = LVAL(DLEN) = maxdelay*RCI + 0.5;
	now = LVAL(NOW) = 0;
	FPTR(DBUF) = (float *) calloc(dlen, sizeof(float));
#ifdef REALTIME
	x->sp_firstbuf = 1;
	nrv = 1;		/* number of radiation vectors specified */
	x->sp_interpol = 1;	/* interpolation is on by default */
#else
	nrv = (narg - X)/5;	/* number of radiation vectors specified */
#endif
	ns = NAs + 1; 	/* Extra surface is for direct path values */
#ifdef REALTIME /* SDY should fix for cmusic */
	FPTR(CARRAYO) = (float *) fp3alloc(nrv,ns,Nchan);
	FPTR(DARRAYO) = (float *) fp3alloc(nrv,ns,Nchan);
	FPTR(AARRAYO) = (float *) fp3alloc(nrv,ns,Nchan);
#endif
	FPTR(CARRAY) = (float *) fp3alloc(nrv,ns,Nchan);
	FPTR(DARRAY) = (float *) fp3alloc(nrv,ns,Nchan);
	FPTR(AARRAY) = (float *) fp3alloc(nrv,ns,Nchan);
	FPTR(FARRAY) = (float *) fp3alloc(nrv,ns,Nchan);
	FPTR(XS) = (float *) calloc(nrv, sizeof(float));
	FPTR(YS) = (float *) calloc(nrv, sizeof(float));
	FPTR(THETAS) = (float *) calloc(nrv, sizeof(float));
	FPTR(AMPS) = (float *) calloc(nrv, sizeof(float));
	FPTR(BACKS) = (float *) calloc(nrv, sizeof(float));

}

void
#ifndef REALTIME
space_free(narg, ap)
register long narg;
register union arg_ptr ap[];
#else
space_free(t_space_tilde *x)
#endif
{
	float *dbuf;
	float ***cut, ***delay, ***atten, ***fader, delsamp;
	long dlen, now, j, nrv, c, r, s, ns;
	float *xs, *ys, *ts, *as, *bs;
	float NAs;
	t_space_room *rm;

	rm = x->sp_room;
	NAs = rm->spr_NAs;
	dbuf = FPTR(DBUF);
	cut = (float ***) FPTR(CARRAY);
	delay = (float ***) FPTR(DARRAY);
	atten = (float ***) FPTR(AARRAY);
	fader = (float ***) FPTR(FARRAY);
#ifdef REALTIME
	nrv = 1;		/* number of radiation vectors specified */
#else
	nrv = (narg - X)/5;	/* number of radiation vectors specified */
#endif
	xs = FPTR(XS);
	ys = FPTR(YS);
	ts = FPTR(THETAS);
	as = FPTR(AMPS);
	bs = FPTR(BACKS);
	ns = NAs + 1;

	free(dbuf);
#ifdef REALTIME /* SDY should fix for cmusic */
	free(x->sp_room);		/* free the rooms */
	fp3free((float ***) FPTR(CARRAYO),nrv,ns,Nchan);
	fp3free((float ***) FPTR(DARRAYO),nrv,ns,Nchan);
	fp3free((float ***) FPTR(AARRAYO),nrv,ns,Nchan);
#endif
	fp3free(cut,nrv,ns,Nchan);
	fp3free(delay,nrv,ns,Nchan);
	fp3free(atten,nrv,ns,Nchan);
	fp3free(fader,nrv,ns,Nchan);
	free(xs);
	free(ys);
	free(ts);
	free(as);
	free(bs);
}

int
space_mkroom(t_space_tilde *x)
{
	int i, j;
	float innersize;
	double dx, dy;

	if (!x->sp_nir) {
		post("space_mkroom: number of rooms not set");
		return (1);
	}
	if (x->sp_outersize < x->sp_innersize) {
		post("space_mkroom: (%f > %f) inner room cannot be larger than outer room", x->sp_innersize, x->sp_outersize);
		return (1);
	}
	x->sp_room = (t_space_room *) calloc(x->sp_nir, sizeof (t_space_room));
	if (!x->sp_room) {
		post("space_mkroom: no memory, calloc() failed");
		return (1);
	}
	x->sp_cr_idx  = 0;

	/*
	 * to keep the old code happy surface corner coordinates are defined
	 * as:
	 *            2    1
	 *
	 *            3    4
	 *
	 * but to keep stereo configuration simply as left and right 
	 * and keep the option of defining more that 4 speakers
	 * speakers coordinates are defined as:
	 *            1    2
	 *
	 *            3    4
	 *
	 */
	for (i = 0; i < x->sp_nir; i++) {
		innersize = x->sp_innersize * (x->sp_nir - i) / x->sp_nir;
		x->sp_room[i].spr_TH = innersize / 4.0;
		x->sp_room[i].spr_CF = SPR_DEFAULT_CF;
		x->sp_room[i].spr_NLs = 4;
		x->sp_room[i].spr_NAs = 4;
		x->sp_room[i].spr_Lx[0] =   innersize / 2;
		x->sp_room[i].spr_Ly[0] =   innersize / 2;
		x->sp_room[i].spr_Lx[1] = - innersize / 2;
		x->sp_room[i].spr_Ly[1] =   innersize / 2;
		x->sp_room[i].spr_Lx[2] = - innersize / 2;
		x->sp_room[i].spr_Ly[2] = - innersize / 2;
		x->sp_room[i].spr_Lx[3] =   innersize / 2;
		x->sp_room[i].spr_Ly[3] = - innersize / 2;

		x->sp_room[i].spr_Ax[0] =   x->sp_outersize / 2;
		x->sp_room[i].spr_Ay[0] =   x->sp_outersize / 2;
		x->sp_room[i].spr_Ax[1] = - x->sp_outersize / 2;
		x->sp_room[i].spr_Ay[1] =   x->sp_outersize / 2;
		x->sp_room[i].spr_Ax[2] = - x->sp_outersize / 2;
		x->sp_room[i].spr_Ay[2] = - x->sp_outersize / 2;
		x->sp_room[i].spr_Ax[3] =   x->sp_outersize / 2;
		x->sp_room[i].spr_Ay[3] = - x->sp_outersize / 2;
		
		switch (x->sp_nchan) {
		case 1:
			x->sp_room[i].spr_ns = 1;
			x->sp_room[i].spr_Sx[0] = 0;
			x->sp_room[i].spr_Sy[0] = innersize / 2;
			break;
		case 2:
			x->sp_room[i].spr_ns = 2;
			x->sp_room[i].spr_Sx[0] = - innersize / 2;
			x->sp_room[i].spr_Sy[0] =   innersize / 2;
			x->sp_room[i].spr_Sx[1] =   innersize / 2;
			x->sp_room[i].spr_Sy[1] =   innersize / 2;
			break;
		case 4:
			x->sp_room[i].spr_ns = 4;
			x->sp_room[i].spr_Sx[0] = - innersize / 2;
			x->sp_room[i].spr_Sy[0] =   innersize / 2;
			x->sp_room[i].spr_Sx[1] =   innersize / 2;
			x->sp_room[i].spr_Sy[1] =   innersize / 2;
			x->sp_room[i].spr_Sx[2] = - innersize / 2;
			x->sp_room[i].spr_Sy[2] = - innersize / 2;
			x->sp_room[i].spr_Sx[3] =   innersize / 2;
			x->sp_room[i].spr_Sy[3] = - innersize / 2;
			break;
		defautl:
			free (x->sp_room);
			post ("space: only mono, stereo and quad configuration supported for now");
			return (1);
		}
		/* set the delay values for the inner rooms */
		/* speakers get phantom images on every inner room */
		for (j = 0; j < x->sp_room[i].spr_ns; j++) {
/* SDY
			dx = x->sp_room[0].spr_Sx[j] - x->sp_room[i].spr_Sx[j];
			dy = x->sp_room[0].spr_Sy[j] - x->sp_room[i].spr_Sy[j];
*/
			dx = x->sp_room[i].spr_Sx[j];
			dy = x->sp_room[i].spr_Sy[j];
			x->sp_room[i].spr_Sdelay[j] = sqrt(
						(double) (dx*dx + dy*dy));
		}
	}
	return (0);
}



/*
 * sp_valcopy -- copy the content of a three dimentional array
 */
void
sp_valcopy(float ***dest, float ***src, int n1, int n2, int n3)
{
	int i, j, k;
	for(i = 0; i < n1; i++)
		for(j = 0; j < n2; j++)
			for(k = 0; k < n3; k++)
				dest[i][j][k] = src[i][j][k] ;
}

#define y1	sp_y1	/* to avoid an overshadowing of the math fun y1() */

void
space
UGHEAD
{
    UGINIT;
    float *Out = Outblock, *Grev = Grevblock;
    float *dbuf;
    float ***cut, ***delay, ***atten, ***fader, delsamp;
    float ***oldcut, ***olddelay, ***oldatten;
    float *xs, *ys, *ts, *as, *bs;
    float x1, y1, x2, y2, t1, t2, fff;
    float d, di, chanval[4], radiant;
    float newatt; /* the new attenuation or interpolated */
    float newcut; /* the new cut value (or interpolated) */
    int place, change, inside = 0;
    long dlen, now, j, nrv, c, r, s = 0, ns;
    float fade, rx, ry, dx, dy, dt, phi, val, tsn;
    float RCI = CI*Srate;
    int t;
    int roomswitch;
    t_space_room *cur_room;

#ifndef REALTIME
    if(STARTNOTE){
	space_init(narg, ap);
    }
#endif
    dbuf = FPTR(DBUF);
    dlen = LVAL(DLEN);
    now = LVAL(NOW);
    cut = (float ***) FPTR(CARRAY);
    delay = (float ***) FPTR(DARRAY);
    atten = (float ***) FPTR(AARRAY);
    fader = (float ***) FPTR(FARRAY);
#ifdef REALTIME
    oldcut = (float ***) FPTR(CARRAYO);
    olddelay = (float ***) FPTR(DARRAYO);
    oldatten = (float ***) FPTR(AARRAYO);
    nrv = 1;		/* number of radiation vectors specified */
#else
    nrv = (narg - X)/5;	/* number of radiation vectors specified */
#endif
    xs = FPTR(XS);
    ys = FPTR(YS);
    ts = FPTR(THETAS);
    as = FPTR(AMPS);
    bs = FPTR(BACKS);
    fade = 1000./(TAU*Srate);
    cur_room = &x->sp_room[x->sp_cr_idx];
    ns = cur_room->spr_NAs + 1;
    /*
     * if we are interpolating the save values are used as one side of the
     * interpolation line
     * and the last number of the current buffer is used as the other side
     * of interpolation value.
     */
    change = 0;
    UGLOOP{
#ifndef REALTIME
	change = (i==0) ;
	dbuf[now] = VAL(0); /* install current sample in delay buffer */
	for(j=0; j<nrv; j++){
	    place = j*5;
	    if(xs[j] != VAL(X+place)){change=1; xs[j]=VAL(X+place);}
	    if(ys[j] != VAL(Y+place)){change=1; ys[j]=VAL(Y+place);}
	    if(ts[j] != VAL(THETA+place)){change=1; ts[j]=VAL(THETA+place);}
	    if(as[j] != VAL(AMP+place)){change=1; as[j]=VAL(AMP+place);}
	    if(bs[j] != VAL(BACK+place)){change=1; bs[j]= VAL(BACK+place);}
	}
#else /* ifndef REALTIME */
	if (x->sp_interpol) {
		if (x->sp_firstbuf) {
			x->sp_firstbuf = 0;
			change = 1;
		}
	} else
		change = (i==0);
	/* i is the index of our pass in the buffers and is defined in UGLOOP */
	dbuf[now] = x->sp_in[i]; /* install current sample in delay buffer */
	j = 0; /* only one nrv */
	/*
	 * we only support a single vector
	 * if this needs to be changed, which is unlikley since one can
	 * start multiple space~ objects in PD, then the way these varialbles
	 * are accessed need to be changed
	 *
	 * SDY CHANGE
	 * it seems that the change in x and y does not need to be
	 * coupled with the change in theta, amp, and back
	 * everytime x and y changes we have to recalcuate everything
	 * but when theta, amp, or back changes we do not need to recalculate
	 * the delay buffers or the cuts
	 */
	if (!x->sp_interpol) {
		if(xs[j] != VAL(X)[i]){change=1; xs[j]=VAL(X)[i];}
		if(ys[j] != VAL(Y)[i]){change=1; ys[j]=VAL(Y)[i];}
		if (x->sp_realray) {
			if(ts[j] != VAL(THETA)[i]) {
				change = 1;
				ts[j] = VAL(THETA)[i];
			}
			if(bs[j] != VAL(BACK)[i]) {
				change=1;
				bs[j]= VAL(BACK)[i];
			}
		} else {
			if (ts[j] != x->sp_theta) {
				change=1;
				ts[j] = x->sp_theta;
			}
			if (bs[j] != x->sp_back) {
				change=1;
				bs[j] = x->sp_back;
			}
		}
		if(as[j] != VAL(AMP)[i]){change=1; as[j]=VAL(AMP)[i];}
	} else if (i == 0) {
		/*
		 * if we are interpolating use the last value of the
		 * buffer as the current value
		 */
		if (xs[j] != VAL(X)[x->sp_vsize - 1]) {
			change = 1;
			xs[j] = VAL(X)[x->sp_vsize - 1];
		}
		if (ys[j] != VAL(Y)[x->sp_vsize - 1]) {
			change = 1;
			ys[j] = VAL(Y)[x->sp_vsize - 1];
		}
		if (x->sp_realray) {
			if (ts[j] != VAL(THETA)[x->sp_vsize - 1]) {
				change = 1;
				ts[j] = VAL(THETA)[x->sp_vsize - 1];
			}
			if (bs[j] != VAL(BACK)[x->sp_vsize - 1]) {
				change = 1;
				bs[j] =  VAL(BACK)[x->sp_vsize - 1];
			}
		} else {
			if (ts[j] != x->sp_theta) {
				change=1;
				ts[j] = x->sp_theta;
			}
			if (bs[j] != x->sp_back) {
				change=1;
				bs[j] = x->sp_back;
			}
		}
		if (as[j] != VAL(AMP)[x->sp_vsize - 1]) {
			change = 1;
			as[j] = VAL(AMP)[x->sp_vsize - 1];
		}
		if (change) {
			sp_valcopy(oldcut,   cut, j + 1, ns, Nchan);
			sp_valcopy(olddelay, delay, j + 1, ns, Nchan);
			sp_valcopy(oldatten, atten, j + 1, ns, Nchan);
		}
	}
#endif /* ifndef REALTIME */
/*
 * get the cut, delay, and atten arrays computed for all current
 * radiation vectors, surface reflection (and direct) paths, and 
 * channels
 */
/* do only on first loop or when something varies */
/* do direct paths first */
	if(change && (!x->sp_interpol || i == 0)){
		/*
		 * determine if we have changed room
		 *
		 * NOTE-SDY
		 * This code assumes that there is only one radiation
		 * vector, both in the interpolation code and
		 * also in its determination
		 * of which inner room we are in
		 */
		j = x->sp_cr_idx;
		roomswitch = 0; /* have we switched in our room calculation */
		if (within(&x->sp_room[j], xs[0], ys[0])) {
			while (j < x->sp_nir - 1) {
				j++;
				if (within(&x->sp_room[j], xs[0], ys[0])) 
					continue;
				else
					break;
			}

		} else {
			while (j) {
				j--;
				if (within(&x->sp_room[j], xs[0], ys[0])) {
					j++;
					break;
				} else
					continue;
				}

		}
		x->sp_cr_idx = j;
		cur_room = &x->sp_room[j];
				
		for(r=0; r<nrv; r++) for(c=0; c<Nchan; c++){
			x1 = xs[r]; y1 = ys[r];
			x2 = cur_room->spr_Sx[c]; y2 = cur_room->spr_Sy[c];
if (c == 1)
			cut[r][0][c] = ngetcut(cur_room, x1,y1,x2,y2,1);
else
			cut[r][0][c] = getcut(cur_room, x1,y1,x2,y2,1);
			dx = x2 - x1 ;
			dx *= dx ;
			dy = y2 - y1 ;
			dy *= dy ;
			d = sqrt((double)(dx + dy)) + cur_room->spr_Sdelay[c];
			delay[r][0][c] = d*RCI;
/* SDY */
if (0 && c == 1) {
	post ("New direct cut= %f", cut[r][0][c]);
	post ("New direct delay = %f", delay[r][0][c]);
	post("cur_room->spr_Sdelay[c] = %f", cur_room->spr_Sdelay[c]);
}
			if ( x->sp_direct != 1. )
				d = pow( (double) d, (double) x->sp_direct ) ;
			if(bs[r] < 1.0){
				dy = y1 - y2;
				dx = x1 - x2;
				phi = dx ? 
				atan2( (double) dy, (double) dx) : 
							(dy > 0 ? D270 : D90 );
				tsn = ts[r];
//SDY				if(tsn > PI)tsn = PI2 - tsn;
				tsn = fmod(tsn, PI);
				dt = tsn - phi;
				dt = dt >= 0. ? dt : -dt;

/*
SDY
				if(dt > PI)dt = PI2 - dt;
				radiant = 1. + (sqrt( (double)(bs[r]) - 1.))
									*dt*IPI;
*/
				dt = fmod(dt, PI);
				radiant = 1. + (bs[r] - 1.) * dt* IPI;
				radiant *= radiant;
				/*
				 * attenuation depends on amplitude of vector,
				 * angle of vector, and distance to vector
				 * origin
				 */
				atten[r][0][c] = as[r]*radiant/(1. + d) ;
			} else
				atten[r][0][c] = as[r]/(1. + d) ;
		}
		/* now do surface reflections */
		for(r=0; r<nrv; r++) for(s=1; s<ns; s++) for(c=0; c<Nchan; c++){
			x1 = xs[r]; y1 = ys[r];
			x2 = cur_room->spr_Sx[c]; y2 = cur_room->spr_Sy[c];
			getrefl(cur_room, x1,y1,x2,y2,s-1,&rx,&ry);
			cut[r][s][c] = 0.;
			cut[r][s][c] = getcut(cur_room, x1,y1,rx,ry,1);
/* SDY
if (c == 0 && s == 1) {
post("x1 = %f ,y1 = %f ,rx = %f ,ry = %f", x1,y1,rx,ry);
	post ("New 1st** cut= %f", cut[r][s][c]);
}
*/
			if (cut[r][s][c])
				cut[r][s][c]*=getcut(cur_room, rx,ry,x2,y2,1);
/* SDY
if (c == 0 && s == 1)
	post ("New 1st cut= %f", cut[r][s][c]);
*/
			dx = x1 - rx ;
			dx *= dx ;
			dy = y1 - ry ;
			dy *= dy ;
			d = sqrt((double) ( dx + dy )) ;
			dx = x2 - rx ;
			dx *= dx ;
			dy = y2 - ry ;
			dy *= dy ;
			d += sqrt((double) (dx + dy)) + cur_room->spr_Sdelay[c];
			delay[r][s][c] = d*RCI;
/* SDY
if (c == 0 && s == 1)
	post ("New 1st reflect delay = %f", delay[r][s][c]);
*/
			if ( x->sp_reflect != 1. )
				d = pow((double) d, (double) x->sp_reflect) ;
			if(bs[r] < 1.0){
				dy = y1 - ry;
				dx = x1 - rx;
				phi = dx ? atan2(dy,dx) : (dy > 0 ? D270 : D90 );
				tsn = ts[r];
//				if(tsn > PI)tsn = PI2 - tsn;
				tsn = fmod(tsn, PI);
				dt = tsn - phi ;
				dt = ( dt >= 0. ? dt : -dt ) ;
/* SDY
				if(dt > PI)dt = PI2 - dt;
				radiant =1.+(sqrt((double)(bs[r]) - 1.))*dt*IPI;
*/
				dt = fmod(dt, PI);
				radiant = 1. + (( (double) ( bs[r]) - 1. ) )*dt*IPI;
				radiant *= radiant;
				atten[r][s][c] = as[r]*radiant/(1. + d) ;
			} else
				atten[r][s][c] = as[r]/(1. + d) ;
	    }
	}

#ifdef REALTIME
	*Grev = 0;
#endif
	for(c=0; c<Nchan; c++){
	    chanval[c] = 0;
	    for(r=0; r<nrv; r++)
	     for(s=0; s<ns; s++){
#ifdef REALTIME
		if (change && x->sp_interpol) {
			newcut = oldcut[r][s][c] +
			 ((cut[r][s][c]-oldcut[r][s][c]) * i) / x->sp_vsize;
			d = olddelay[r][s][c] +
			 ((delay[r][s][c]-olddelay[r][s][c]) * i) / x->sp_vsize;
			newatt = oldatten[r][s][c] +
			 ((atten[r][s][c]-oldatten[r][s][c]) * i) / x->sp_vsize;
		} else {
			newcut = cut[r][s][c];
			newatt =  atten[r][s][c];
			d = delay[r][s][c];
		}
#else
		newatt =  atten[r][s][c];
		d = delay[r][s][c];
#endif
		delsamp = do_delay(dbuf,dlen,now,d);

		if(s == 0 && (c == 0 || r != 0)){
		    if(!within(cur_room, xs[r], ys[r])){  /* Outside? */
			*Grev += delsamp * newatt;
			inside = 0;
		    } else {
			inside = 1;
		    }
		} else if(!inside) 
			*Grev += delsamp * newatt;
		chanval[c] +=  delsamp * newatt * newcut;
	    }
	    di = chanval[c];
#ifndef REALTIME
	    *Out++ += di ;
#else /* ifndef REALTIME */
	    ((float **)Out)[c][i] = di;
#endif
	    val = ( di >= 0 ? di : -di ) ;
	    if(val > Maxecho)Maxecho = di ;
	}
	Grev++;
	now -= 1;
	if(now < 0)now = dlen - 1;
	LVAL(NOW) = now;
	LVAL(DLEN) = dlen;
	UGEND(0);
    }
#ifndef REALTIME
    if(ENDNOTE){
	space_free(narg, ap);
    }
#endif
}



#define	DIFFRACTION(x);							\
		if (x < rm->spr_TH)					\
			newx = (pow((rm->spr_TH-x)/rm->spr_TH,1./rm->spr_CF));	\
		else							\
			newx = 0;					\


float
ngetcut(t_space_room *rm, float x1, float y1, float x2, float y2, int retfrac)
{
return(getcut(rm, x1, y1, x2, y2, retfrac));
}

/*
 * getcut returns 0.0 if the line defined by x1,x2 and x2,y2 crosses 
 * (cuts) any surface of the inner room, and 1.0 if it doesn't
 *
 * if retfrac is 1 the algorthm returns a fractional cut value
 */
float
getcut(t_space_room *rm, float x1, float y1, float x2, float y2, int retfrac)
{
	register int i;
	int test1, test2 ;
	float t1, t2 ;
	float m_ray = 0, b_ray = 0;
	int horiz_ray=0, vert_ray=0, first_ray=1;
	int cutthrough;	/* if we cut through speaker locations */
	float *lx1, *ly1;
	float *lx2, *ly2;
	int *horiz, *vert;
	float *m, *b;
	float x, newx = 0;
	float ref_factor;
	float y;
	float xsection;
	float ysection;

	lx1 = rm->spr_lx1;
	lx2 = rm->spr_lx2;
	ly1 = rm->spr_ly1;
	ly2 = rm->spr_ly2;
	horiz = rm->spr_vert;
	vert = rm->spr_horiz;
	m = rm->spr_m;
	b = rm->spr_b;


    if (!rm->spr_notfirst) {
	if (SP_MAX_SURFACE != 4)
		post("space: getcut() internal error, this routine only works for square rooms");
	/*
	 * precompute equations for walls
	 */
	rm->spr_notfirst = 1 ;
	for( i=0; i<rm->spr_NLs; i++ ) {
	    lx1[i] = rm->spr_Lx[i];
	    ly1[i] = rm->spr_Ly[i];
	    lx2[i] = rm->spr_Lx[(i+1)%rm->spr_NLs] ;
	    ly2[i] = rm->spr_Ly[(i+1)%rm->spr_NLs] ;
		if (ly1[i] == ly2[i]) {
			horiz[i] = 1;
			continue;
		} else
			horiz[i] = 0;
		if (lx1[i] == lx2[i]) {
			vert[i] = 1;
			continue;
		} else
			vert[i] = 0 ;
		
		m[i] = (ly2[i] - ly1[i]) / (lx2[i] - lx1[i]);
		b[i] = ly1[i] - (m[i] * lx1[i]);
	}
    }


    cutthrough = 0;
    ref_factor = 1.0;
    for (i=0; i<rm->spr_NLs; i++) {
	if( horiz[i] ) {
		/*
		 * if one end of ray touches wall, no cut possible
		 */
		if (y1 == ly1[i])
			continue;
		if (y2 == ly2[i])
			continue;
		test1 = y1 > ly1[i];
		test2 = y2 > ly2[i];
	} else if( vert[i] ) {
		if (x1 == lx1[i])
			continue;
		if (x2 == lx2[i])
			continue;
		test1 = x1 > lx1[i];
		test2 = x2 > lx2[i];
	} else {

		t1 = (m[i] * x1) + b[i];
		if( t1 == y1 )
			continue;
		test1 = t1 > y1;
		t2 = (m[i] * x2) + b[i];
		if( t2 == y2 )
			continue;
		test2 = t2 > y2;
	}
/*
 * if both ray endpoints on same side of wall, no cut possible
 */
	if( test1 == test2 ) continue;
/*
 * come here iff ray endpoints on opposite sides of (infinitely extended) wall
 * compute normal equation of ray
 */
	if(first_ray)
	{
		first_ray=0;
		if (y1 == y2)
			horiz_ray = 1;
		else if (x1 == x2)
			vert_ray = 1;
		else {
			m_ray = (y2 - y1) / (x2 - x1);
			b_ray = y1 - (m_ray * x1);
		}
	}

	if(horiz_ray) {
		if (y1 == ly1[i])
			continue;
		if (y2 == ly2[i])
			continue;
		test1 = ly1[i] > y1;
		test2 = ly2[i] > y1;
	} else if(vert_ray) {
		if (x1 == lx1[i])
			continue;
		if (x2 == lx2[i])
			continue;
		test1 = lx1[i] > x1;
		test2 = lx2[i] > x1;
	} else {
		t1 = (m_ray * lx1[i]) + b_ray;
/* SDY
		if (t1 == ly1[i])
			continue;
*/
		/*
		 * if we touch two end points of two different walls
		 * we must be going through the room
		 */
		if (t1 == ly1[i]) {
//NEW CODE SDY
// The following is a simple version of what should be here
// There should be code to look at the size of both walls we are going through
// in comparison to the TH value
			if (cutthrough) {
				if (horiz[i]) {
					DIFFRACTION(fabs(lx1[i] - lx2[i]));
				} else if (vert[i]) {
					DIFFRACTION(fabs(ly1[i] - ly2[i]));
				} else  {
					DIFFRACTION(sqrt(pow(lx1[i]-lx2[i],2) +
							pow(ly1[i]-ly2[i], 2)));
				}
				return(newx);
			}
			cutthrough = 1;
			continue;
		}
		test1 = t1 > ly1[i];
		t2 = (m_ray * lx2[i]) + b_ray;
/* SDY
		if (t2 == ly2[i])
			continue;
*/
		if (t2 == ly2[i]) {
//NEW CODE SDY
			if (cutthrough) {
				if (horiz[i]) {
					DIFFRACTION(fabs(lx1[i] - lx2[i]));
				} else if (vert[i]) {
					DIFFRACTION(fabs(ly1[i] - ly2[i]));
				} else  {
					DIFFRACTION(sqrt(pow(lx1[i]-lx2[i],2) +
							pow(ly1[i]-ly2[i], 2)));
				}
				return(newx);

				continue;
			}
			cutthrough = 1;
			continue;
		}
		test2 = t2 > ly2[i];
	}
/*************************************************************************/
	if( test1 != test2) {
		if (horiz[i]) {
			if (m_ray < 0) {
				if (m_ray >= -10)
					x = 0;
				else {
					x =  1 - 1/pow(-(m_ray + 9), 1./3);
				}
				xsection = (ly1[i] - b_ray) / m_ray;
				t2 = fabs(lx1[i] - xsection);
				t1 = fabs(xsection - lx2[i]);
				DIFFRACTION(t2);
			} else if (m_ray > 0) {
				if (m_ray <= 10)
					x = 0;
				else
					x =  1 - 1/pow(m_ray - 9, 1./3);
				xsection = (ly1[i] - b_ray) / m_ray;
				t2 = fabs(lx1[i] - xsection);
				t1 = fabs(xsection - lx2[i]);
				DIFFRACTION(t1);
			} else {
				x = 1;
		printf("bad case %3.3f,%3.3f - %3.3f,%3.3f -- i = %d mray = %3.3f\n",
								x1, y1, x2, y2, i, m_ray);
			}
		} else if (vert[i]) {
			if (m_ray < 0) {
				if (m_ray <= -.1)
					x = 0;
				else
					x = 1 - pow(-(m_ray * 10), 1./3);
				ysection = (lx1[i] * m_ray) + b_ray;
				t2 = fabs(ly1[i] - ysection);
				t1 = fabs(ysection - ly2[i]);
				DIFFRACTION(t1);
			} else if (m_ray > 0) {
				if (m_ray >= .1)
					x = 0;
				else
					x = 1 - pow(10 * m_ray, 1./3);
				ysection = (lx1[i] * m_ray) + b_ray;
				t2 = fabs(ly1[i] - ysection);
				t1 = fabs(ysection - ly2[i]);
				DIFFRACTION(t2);
			} else {
				x = 1;
				printf("bad case2 %3.3f,%3.3f - %3.3f,%3.3f -- i = %d mray = %3.3f\n",
								x1, y1, x2, y2, i, m_ray);
			}
		} else {
			printf("bad case Neither vertical nor horizontal\n");
			printf("bad case %3.3f,%3.3f - %3.3f,%3.3f -- i = %d mray = %3.3f\n",
								x1, y1, x2, y2, i, m_ray);
		}

		if (newx < ref_factor) {
			ref_factor = newx;
		}
		if (ref_factor < 0.00001)
			return (ref_factor);
	}
	continue;
		
/*************************************************************************/
/*
 * if wall endpoints also on opposite sides of ray, then cut
 */
	if( test1 != test2 )
	    return( 0. );	/* ray path cut */
    }
    return (ref_factor);	/* ray path not cut */
}

/*
 * within return 1 if point x,y is inside listening space, else 0
 */
int
within(t_space_room *rm, float x,float y)
{
	if(x < rm->spr_Lx[0] && y < rm->spr_Ly[0])
		if(x > rm->spr_Lx[1] && y < rm->spr_Ly[1])
			if(x > rm->spr_Lx[2] && y > rm->spr_Ly[2])
				if(x < rm->spr_Lx[3] && y > rm->spr_Ly[3])
					return(1);
	return(0);
}

/*
 * getrefl calculates the reflection point (rx,ry) of a ray path from
 * source point (sx,sy) to surface s and back to channel output at (cx,cy)
 */
void
getrefl(t_space_room *rm, float sx, float sy, float cx, float cy,
						int s, float *rx, float *ry)
{
	double lx1,lx2,ly1,ly2,d2,s2,c2,r2,x,y,a;
	double dx1, dy1, dx2, dy2, m1 = 0, m2 = 0, b1 = 0, b2 = 0;
	double *d, *sint, *cost, *rho, *sin2t, *cos2t, *f1, *f2;

	if (SP_MAX_SURFACE != 4)
		post("space: getrefl() internal error, this routine only works for square rooms");
	d = rm->spr_d;
	sint = rm->spr_sint;
	cost = rm->spr_cost;
	rho = rm->spr_rho;
	sin2t = rm->spr_sin2t;
	cos2t = rm->spr_cos2t;
	f1 = rm->spr_f1;
	f2 = rm->spr_f2;

	s = s%rm->spr_NAs;
	lx1 = rm->spr_Ax[s];
	ly1 = rm->spr_Ay[s];
	lx2 = rm->spr_Ax[(s+1)%rm->spr_NAs];
	ly2 = rm->spr_Ay[(s+1)%rm->spr_NAs];
	/*
	* find normal equations for walls (needs to be done only once)
	*/
	if(d[s] == 0.){	
		d[s] = 1./sqrt((double) ((lx2-lx1)*(lx2-lx1) +
							(ly2-ly1)*(ly2-ly1)));
		sint[s] = (ly2-ly1)*d[s];
		cost[s] = (lx2-lx1)*d[s];
		rho[s] = ly1*cost[s] - lx1*sint[s];
		if(rho[s]>0.){
			sint[s]= -sint[s];
			cost[s]= -cost[s];
			rho[s]= -rho[s];
		}
		sin2t[s] = 2.*sint[s]*cost[s];
		cos2t[s] = 2.*cost[s]*cost[s] - 1.;
		f1[s] = -2.*rho[s]*sint[s];
		f2[s] =  2.*rho[s]*cost[s];
	}
	/*
	 * if the source is behind the surface for which we are calculating
	 * a phantom source, assume the source to be the phantom
	 */
	if (lx1 > 0 && lx2 > 0) {
		if (cx > lx1) {
			*rx = cx;
			*ry = cy;
			return;
		}
	} else if (lx1 < 0 && lx2 < 0) {
		if (cx < lx1) {
			*rx = cx;
			*ry = cy;
			return;
		}
	} else if (ly1 > 0 && ly2 > 0) {
		if (cy > ly1) {
			*rx = cx;
			*ry = cy;
			return;
		}
	} else if (ly1 < 0 && ly2 < 0) {
		if (cy < ly1) {
			*rx = cx;
			*ry = cy;
			return;
		}
	} else {
/* SDY */
post("what is this case?");
	}
	/*
	* find phantom source location
	*/
	x = f1[s] + sy*sin2t[s] + sx*cos2t[s];
	y = f2[s] - sy*cos2t[s] + sx*sin2t[s];
	/*
	* calculate and return wall intercept of ray from phantom source to speaker
	*/
	if( (dx1 = x - cx) != 0.){ 
		m1 = (y - cy)/dx1; 
		b1 = y - m1*x; 
	}
	if( (dx2 = lx2 - lx1) != 0.){ 
		m2 = (ly2 - ly1)/dx2; 
		b2 = ly1 - m2*lx1; 
	} else { 
		*ry = m1*lx1 + b1; 
		*rx = lx1; 
		return; 
	}
	if(dx1 == 0.){ 
		*ry = m2*x + b2; 
		*rx = cx; 
		return; 
	}
	*rx = (b2 - b1)/(m1 - m2); 
	*ry = m1*(*rx) + b1; 
	return;

/*
 * alternative method using normal equations (slower)
 */
/*
 *     d2 = 1./sqrt( (double) ( (x-cx)*(x-cx) + (y-cy)*(y-cy) ) );
 *     s2 = (y-cy)*d2;
 *     c2 = (x-cx)*d2;
 *     r2 = y*c2 - x*s2;
 *     if(r2>0.){s2= -s2; c2= -c2; r2= -r2;}
 *     a = 1./(sint[s]*c2 - cost[s]*s2);
 *     dy1 = (r2*sint[s] - rho[s]*s2)*a;
 *     dx1 = (r2*cost[s] - rho[s]*c2)*a;
 */
}
/*
 * do_delay looks up a sample delayed by tau samples using interpolation
 * on a circular buffer of length len with present sample at buf[now]
 *
 * 4 point interpolation borrowed from pd sources
 */
float
do_delay(buf,len,now,tau)
float buf[]; float tau; long len, now;
{
	register long t1, t2 ;
	float frac,  a,  b,  c,  d, cminusb;
	float *fp;


	if (!len) {
		fprintf(stderr,"\nspace unit: do_delay internal error.\n");
		return (0.0);
	}
	t1 = now + tau;
	if (t1 >= len)
		t1 -=len;
//	if (t1 >= len) {
//		fprintf(stderr,"\nspace unit: do_delay internal error.\n");
//		fprintf(stderr,"\nspace unit: t1 = %d, len = %d\n",
//							(int)t1, (int)len);
//		fprintf(stderr,"\nspace unit: tau = %f, now = %d\n",
//							tau, (int) now);
//		return (0.0);
//	}
	while(t1 >= len)t1 -= len; /* old code --SDY */
	t2 = t1 + 1;
	if (t2 >= len)
		t2 -=len;
	if (t2 >= len) {
		fprintf(stderr,"\nspace unit: do_delay internal error.\n");
		fprintf(stderr,"\nspace unit: t2 = %d, len = %d\n",
							(int) t2, (int) len);
		return (0.0);
	}
	while(t2 >= len)t2 -= len; /* old code --SDY */

	switch (space_delinterpol) {
	case 0: /* linear interpolation */
		break;
	case 1:
		if (t1 + 2 > len || t1 < 1)
			break;
		frac = t1 - ((int)t1);
		fp = buf + t1;
		a = fp[-1];
		b = fp[0];
		c = fp[1];
		d = fp[2];
		cminusb = c-b;
		return (b + frac * (
			    cminusb - 0.5f * (frac-1.) * (
			(a - d + 3.0f * cminusb) * frac + (b - a - cminusb)
			)));
	default:
		fprintf(stderr,"\nspace: do_delay unknown interpolation.\n");
		break;
	}
	return(buf[t1] + (tau - ( (int) tau ))*(buf[t2] - buf[t1]));
}
/*
 * fp3alloc allocates memory for a 3-D array, C-style
 */
float *** fp3alloc(x,y,z) register int x,y,z; {
 register float ***fp3; register int i,j;
    if ((fp3 = (float ***) calloc(x,sizeof(float **))) == NULL){
	fprintf(stderr,"\nCMUSIC: space generator: fp3alloc failed.\n");
	exit (-1);
    }
    for (i = 0; i < x; i++) {
	if ((fp3[i] = (float **) calloc(y, sizeof(float *))) == NULL){
	    fprintf(stderr,"\nCMUSIC: space generator: fp3alloc failed.\n");
	    exit (-1);
	}
	for (j = 0; j < y; j++) {
	    if ((fp3[i][j] = (float *) calloc(z, sizeof(float))) == NULL){
		fprintf(stderr,"\nCMUSIC: space generator: fp3alloc failed.\n");
		exit (-1);
	    }
	}
    }
    return(fp3);
}
/*
 * fp3free frees memory for a 3-D array, C-style
 */
void
fp3free(fp3,x,y,z) register float ***fp3; register int x,y,z; {
 register int i,j;
    for (i = 0; i < x; i++) {
	for (j = 0; j < y; j++) {
	    free(fp3[i][j]);
	}
	free(fp3[i]);
    }
    free(fp3);
}

#ifdef notdef /*old code from cmusic */
/*
 * getcut returns 0.0 if the line defined by x1,x2 and x2,y2 crosses 
 * (cuts) any surface of the inner room, and 1.0 if it doesn't
 */
float oldgetcut(x1,y1,x2,y2, fracret)
float x1,y1,x2,y2; int fracret; {
 register int i;
 int test1, test2 ;
 float dx, dy, dinv, t1, t2, lsintheta, lcostheta, lrho ;
 static float costheta[4], sintheta[4], rho[4] ;
 static float lx1[4], ly1[4], lx2[4], ly2[4] ;
 static int horiz[4], vert[4] ;
 static int first = 1 ;

 extern int Newgetcut;
if (Newgetcut)
	return (ngetcut(x1,y1,x2,y2, fracret));

    if( first ) {
/*
 * precompute normal equations for walls
 */
	first = 0 ;
	for( i=0; i<NLs; i++ ) {
	    lx1[i] = Lx[i] ;
	    ly1[i] = Ly[i] ;
	    lx2[i] = Lx[(i+1)%NLs] ;
	    ly2[i] = Ly[(i+1)%NLs] ;
	    if( ly1[i] == ly2[i] ) horiz[i] = 1 ; else horiz[i] = 0 ;
	    if( lx1[i] == lx2[i] ) vert[i] = 1 ; else vert[i] = 0 ;
	    dx = lx2[i] - lx1[i] ;
	    dy = ly2[i] - ly1[i] ;
	    dinv = 1./sqrt( (double) ( dx*dx + dy*dy ) ) ;
	    sintheta[i] = dy*dinv ;
	    costheta[i] = dx*dinv ;
	    rho[i] = ( ly1[i]*lx2[i] - lx1[i]*ly2[i] )*dinv ;
	}
    }
    for( i=0; i<NLs; i++ ) {
	if( horiz[i] ) {
/*
 * if one end of ray touches wall, no cut possible
 */
	    if( y1 == ly1[i] ) continue ;
	    if( y2 == ly2[i] ) continue ;
	    test1 = y1 > ly1[i] ;
	    test2 = y2 > ly2[i] ;
	} else if( vert[i] ) {
	    if( x1 == lx1[i] ) continue ;
	    if( x2 == lx2[i] ) continue ;
	    test1 = x1 > lx1[i] ;
	    test2 = x2 > lx2[i] ;
	} else {
	    t1 = y1*costheta[i] - x1*sintheta[i] ;
	    if( t1 == rho[i] ) continue ;
	    test1 = t1 > rho[i] ;
	    t2 = y2*costheta[i] - x2*sintheta[i] ;
	    if( t2 == rho[i] ) continue ;
	    test2 = t2 > rho[i] ;
	}
/*
 * if both ray endpoints on same side of wall, no cut possible
 */
	if( test1 == test2 ) continue ;
/*
 * come here iff ray endpoints on opposite sides of (infinitely extended) wall
 * compute normal equation of ray
 */
	dx = x2 - x1 ;
	dy = y2 - y1 ;
	dinv = sqrt( (double) ( dx*dx + dy*dy ) ) ;
	if( dinv == 0. ) continue ;
	dinv = 1./dinv ;
	lsintheta = dy*dinv ;
	lcostheta = dx*dinv ;
	lrho = ( y1*x2 - x1*y2 )*dinv ;
	t1 = ly1[i]*lcostheta - lx1[i]*lsintheta ;
/*
 * if wall endpoint touches ray, no cut possible
 */
	if( t1 == lrho ) continue ;
	test1 = t1 > lrho ;
	t2 = ly2[i]*lcostheta - lx2[i]*lsintheta ;
	if( t2 == lrho ) continue ;
	test2 = t2 > lrho ;
/*
 * if wall endpoints also on opposite sides of ray, then cut
 */
	if( test1 != test2 )
	    return( 0. ) ;	/* ray path cut */
    }
    return( 1. ) ;	/* ray path not cut */
}
/**** SDY OLD code now replaced with the diffraction code in getcut()
from the main UGHEAD loop

		if(cut[r][s][c] == 1.){ 
			fff = fader[r][s][c] ;
			if(fff < 1.){
				fff += fade;
				if(fff > 1.)
					fff = 1.;
				val = delsamp * newatt * fff ;
				chanval[c] += val;
				if(inside)
					*Grev += val;
			} else {
				val = delsamp * newatt;
				chanval[c] += val;
				if(inside)
					*Grev += val;
			}
			fader[r][s][c] = fff ;
		} else { 
			fff = fader[r][s][c] ;
			if(fff > 0.){
				fff -= fade;
				if(fff < 0.)
					fff = 0.;
				    val = delsamp * newatt * fff;
				    chanval[c] += val;
				    if(inside)
				    	*Grev += val;
			}
			fader[r][s][c] = fff ;
		}
*******************/

#endif
