/*	%M%	%I%	%H%	*/

# ifdef vax
#include "../conf/ds.h"
# if NDS > 0
# endif vax

#include "../h/param.h"
#include "../h/dir.h"
#include "../h/user.h"
#include "../h/buf.h"
#include "../h/proc.h"
#include "../h/systm.h"
# ifdef vax
#include "../h/map.h"
#include "../h/pte.h"
#include "../h/uba.h"
# endif vax
#include "../h/ds.h"

/*
 * DSC System 200 driver
 * via DSC dma11
 */

struct	dsregs {
	/* dma registers */
	short	dmablr;			/* block length */
	short	dmasax;			/* starting address extension */
	short	dmasar;			/* starting address */
	short	dmacsr;			/* command status */
	short	dmawc;			/* word count */
	short	dmaacx;			/* address count extension */
	short	dmaac;			/* address count */
	short	dmadr;			/* data */
	short	dmaiva;			/* interrupt vector */
	short	dmacls;			/* clear status */
	short	dmaclr;			/* clear device */
	short	unused1;

	/* asc registers */
	short	ascdis;			/* display register */
	short	asccsr;			/* command status register */
	short	ascsrt;			/* sample rate register */
	short	ascrst;			/* reset */
	short	ascseq[16];		/* sequence rams */

	/* converter registers */
	short	adc[8];			/* analog to digital converters */
	short	dac[8];			/* digital to analog converters */
};

/*
 * ASC csr bits
 */
# define RUN	bit(0)			/* run conversion */
# define HZ20	(0)			/* filter, 20kHz */
# define HZ10	bit(1)			/* filter, 10kHz */
# define BYPASS	bit(2)			/* external filter */
# define HZ05	(bit(1)|bit(2))		/* filter, 5kHz */
# define RECORD	bit(3)			/* a/d */
# define PLAY	bit(4)			/* d/a */
# define BRD	bit(5)			/* broadcast */
# define MON	bit(6)			/* monitor */
# define ASCIE	bit(7)			/* interrupt enable */
# define BA	bit(8)			/* bus abort */
# define DCN	bit(9)			/* dc not ok */
# define DNP	bit(12)			/* device not present */
# define DER	bit(13)			/* device error */
# define DLT	bit(14)			/* data late */
# define ERR	bit(15)			/* error */

/*
 * DMA csr bits
 */
# define DRF	bit(0)			/* data register full */
# define AMPS	bit(1)			/* memory port select */
# define AMPE	bit(2)			/* memory parity error */
# define XBA	bit(3)			/* external buss abort */
# define W2M	bit(4)			/* write to memory */
# define CHN	bit(5)			/* chain */
# define DMAIE	bit(6)			/* interrupt enable */
# define BSY	bit(7)			/* busy */
# define SFL	bit(9)			/* starting address register full */
# define OFL	bit(10)			/* offline */
# define XIN	bit(11)			/* external interrupt */
# define IIN	bit(12)			/* internal interrrupt */
# define XER	bit(13)			/* external error */
# define UBA	bit(14)			/* unibus abort */

# ifdef notdef
# define MINARG (A_TODO | A_BNO | A_CNT | A_SEQ)
# endif notdef
# define MINARG (A_BNO | A_CNT | A_SEQ)
# define DSPRI	(PZERO-1)
# define DISK	makedev(13, 2)		/* xylogics controlled cdc 300mb */
# define BLPT	32			/* blocks per track */

# define bit(x)				(1 << (x))
# define HZMSK				(bit(1)|bit(2))
# define block(cnt, todo)		((((todo - cnt) / dsf.f_bsize) * BLPT) + dsf.f_bno)
# define size(cnt)			((cnt > dsf.f_bsize) ? dsf.f_bsize : cnt)
# define diskio(bufp)			xystrategy(bufp)

/*
 * ds states.
 */
# define DSCLOSED	00
# define DSOPEN		01
# define DSBSY		02
# define DSNDSK		040

/*
 * params to driver
 */
# define A_TODO		01
# define A_BNO		02
# define A_CNT		04
# define A_SEQ		010

# define ADBASE		040
# define DABASE		050

int dswatch();

struct buf dsb[NDSB];
int ds_wticks;

# ifdef vax
int dsubinfo[NDSB];
# endif vax

/*
 * Relevant information
 * about the dma and asc.
 */
struct ds {
	short	d_dmacsr;		/* copy of dma csr on error */
	short	d_asccsr;		/* copy of asc csr on error */
	short	d_state;		/* internal flags */
	short	d_errs;			/* errors, returned via ioctl */
	short	d_bufno;		/* dsubinfo/buffer */
	short	d_uid;			/* user id */
	short	d_args;			/* args recieved from user */
} ds;

/*
 * Relevant information
 * about the disking.
 */
struct dsf {
	caddr_t	f_base;			/* base of data (user buffer) */
	daddr_t	f_bno;			/* starting block */
	off_t	f_todo;			/* amnt. of file to convert */
	off_t	f_ccnt;			/* amnt. of data not converted */
	off_t	f_dcnt;			/* amnt. of file not diskio'd */
	off_t	f_icnt;			/* amnt. of file not seen by dsintr */
	int	f_bsize;		/* size of each buffer */
	int	f_boff;			/* offset into buffer of first i/o */
} dsf;

/* ARGSUSED */
dsopen(dev, flag) {
	register struct dsregs *dsr;
	short dummy;

	if (ds.d_state & DSOPEN) {
		u.u_error = ENXIO;	/* in use */
		return;
	}

	/* clear dsf */
	dsf.f_base = 0;
	dsf.f_bno = 0;
	dsf.f_todo = 0;
	dsf.f_ccnt = 0;
	dsf.f_dcnt = 0;
	dsf.f_bsize = 0;
	dsf.f_boff = 0;

	/* clear ds */
	ds.d_dmacsr = 0;
	ds.d_asccsr = 0;
	ds.d_state = 0;
	ds.d_errs = 0;
	ds.d_bufno = 0;
	ds.d_uid = u.u_ruid;
	ds.d_args = 0;

	dsr = DSADDR;

	if (dsr->dmacsr & OFL) {
		u.u_error = EIO;		/* off line */
		return;
	}

	/*
	 * Initialize.
	 */
	dsinit();
	dsr->dmacsr = CHN;
	dsr->asccsr &= ~ RUN;
	dummy = dsr->dmasar;
	dsr->asccsr &= ~ MON;
	dummy = dsr->ascrst;
 	dsr->asccsr |= ASCIE;
	dsr->dmacls = 0;
	dsr->dmacsr |= DMAIE;
	dsr->dmawc = -1;

	/*
	 * If reading then doing a/d,
	 * going to memory then to
	 * disk (record) so set W2M.
	 */
	if (flag == 0) {
		dsr->asccsr |= RECORD;
		dsr->dmacsr |= W2M;
	}
	else
		dsr->asccsr |= PLAY;

	ds.d_state = DSOPEN;
}

dsinit() {
	register struct dsregs *dsr;
	short dummy;

	/*
	 * Possible abnormal termination;
	 * only the current user or root can
	 * terminate an active run.
	 */
	if (ds.d_state & DSOPEN) {
		if (ds.d_uid != u.u_ruid) {
			if (u.u_uid != 0) {
				u.u_error = ENXIO;
				return;
			}
		}
	}

	dsr = DSADDR;

	if (dsr->dmacsr & OFL)
		printf("asc offline\n");
	else {
		dsr->asccsr = 0;
		dsr->ascrst = 0;
	}

	dsr->dmablr = 0;
	dsr->dmasax = 0;
	dsr->dmasar = 0;
	dsr->dmacsr = 0;
	dsr->dmawc = 0;
	dsr->dmaacx = 0;
	dsr->dmaac = 0;
	dsr->dmadr = 0;
	dsr->dmaiva = 0;
	dsr->dmaclr = 0;
	dsr->dmacls = 0;
	dummy = dsr->dmasar;			/* clear sar flag */

	/*
	 * Put converters in monitor mode.
	 */
 	dsseq(0, ADBASE+0, AD);
 	dsseq(1, ADBASE+1, AD);
	dsr->ascseq[1] |= bit(7);		/* last sequence reg */
	dsr->asccsr = RUN | MON;

	/*
	 * Abnormal termination.
	 */
	if (ds.d_state & DSBSY) {
		ds.d_state &= ~ DSBSY;
		wakeup((caddr_t) &ds);
	}
}

/* ARGSUSED */
dsclose(dev, flag) {
	ds.d_state = DSCLOSED;
	dsinit();
}

/*
 * Count is a copy of u.u_count, and is the size of the user buffer, e.g.
 * the 3rd argument to read() or write().
 *
 * Base is a copy of u.u_base and is the address of the user buffer, e.g.
 * the 2nd argument to read() or write().
 *
 * Bsize is the size of the buffers used for the i/o. The buffers are
 * obtained by taking the user buffer and splitting it into NDSB buffers.
 * Currently the user buffer is constrained to be a multiple of the
 * track size of the cdc disk (MINDSB).
 *
 * Using count, base, and bsize each buffer header is set up. The converters
 * only need the base address of the buffer and the word count. The disk needs
 * these in addition to the block number. If we are doing d/a conversions then
 * we start the disk up to fill up the buffers.
 *
 * After everything is ready to go we turn on the RUN bit and let 'em rip.
 */
dsstart(rw) {
	register struct dsregs *dsr;
	unsigned int count;
	caddr_t base;
	caddr_t oaddr;
	off_t bsize;
	int flag;
	int i;
	int ts, nb;

	count = u.u_count;
	bsize = (count / NDSB);
	dsf.f_bsize = bsize;
	base = u.u_base;
	dsr = DSADDR;

	/*
	 * Make sure we have all
	 * necessary info.
	 */
	if ((ds.d_args & MINARG) != MINARG) {
		ds.d_errs |= DSARGS;
		goto bad;
	}

# ifdef vax
	/*
	 * Either the converters will be writing
	 * to memory or the disk will be.
	 */
	if (useracc(base, count, B_WRITE) == NULL) {
		ds.d_errs |= DSACC;
		u.u_error = EFAULT;
		return;
	}
# endif vax

	/*
	 * Check for misaligned buffer.
	 */
	if ((count % MINDSB) != 0) {
		ds.d_errs |= DSMOD;
		goto bad;
	}

	/*
	 * Check for tiny buffer.
	 */
	if (bsize < MINDSB) {
		ds.d_errs |= DSSIZE;
		goto bad;
	}

	/*
	 * Check for unreasonable buffer
	 * offset for first buffer.
	 */
	if (dsf.f_boff >= bsize) {
		ds.d_errs |= DSARGS;
		goto bad;
	}

	flag = u.u_procp->p_flag & SULOCK;
	u.u_procp->p_flag |= SULOCK;

	ds.d_state |= DSBSY;

# ifdef vax
	vslock(base, count);		/* fasten seat belts */
# endif vax

	/*
	 * Point each dsb somewhere into
	 * the user's buffer, set up each dsb.
	 * The rw flag is set to the inverse of
	 * our operation since it applies to the
	 * disk i/o to be done.
	 *
	 * This will need to be rewritten for the
	 * pdp.
	 */
	for (i = 0; i < NDSB; i++) {
# ifdef pdp11
		/*
		 * Check odd base, odd count, and address wraparound
		 */
		if (base&01 || u.u_count&01 || base>=base+u.u_count)
			goto bad;
		ts = (u.u_tsize+127) & ~0177;
		if (u.u_sep)
			ts = 0;
		nb = (base>>6) & 01777;
		/*
		 * Check overlap with text. (ts and nb now
		 * in 64-byte clicks)
		 */
		if (nb < ts)
			goto bad;
		/*
		 * Check that transfer is either entirely in the
		 * data or in the stack: that is, either
		 * the end is in the data or the start is in the stack
		 * (remember wraparound was already checked).
		 */
		if ((((base+u.u_count)>>6)&01777) >= ts+u.u_dsize
		    && nb < 1024-u.u_ssize)
			goto bad;
		/*
		 * Compute physical address by simulating
		 * the segmentation hardware.
		 */
		ts = (u.u_sep? UDSA: UISA)->r[nb>>7] + (nb&0177);
		dsb[i].b_un.b_addr = (caddr_t)((ts<<6)+((base+(i*bsize))&077));
		dsb[i].b_xmem = (ts>>10) & 077;
# endif pdp11
# ifdef vax
		dsubinfo[i] = 0;
		dsb[i].b_un.b_addr = base + (i * bsize);
# endif vax
		dsb[i].b_error = 0;
		dsb[i].b_proc = u.u_procp;
		dsb[i].b_flags = B_PHYS | ((rw == B_READ) ? B_WRITE : B_READ);
		dsb[i].b_dev = DISK;
		dsb[i].b_bcount = bsize;
		dsb[i].b_flags |= B_DSC | B_BUSY;
		if (rw == B_WRITE) {
			dsb[i].b_blkno = block(dsf.f_dcnt, dsf.f_todo);
			if ((ds.d_state & DSNDSK) == 0) {
				diskio(&dsb[i]);
				while ((dsb[i].b_flags & B_DONE) == 0)
					sleep((caddr_t) &dsb[i], DSPRI);
				if (dsb[i].b_flags & B_ERROR) {
					ds.d_state &= ~ DSBSY;
					ds.d_errs |= DSDISK;
					goto out;
				}
			}
			else
				dsb[i].b_flags |= B_DONE;
			dsf.f_dcnt -= bsize;
		}
		else {
			dsb[i].b_flags |= B_DONE;
		}
	}

	/*
	 * Start watchdog.
	 */
	timeout(dswatch, (caddr_t) 0, HZ);

# ifdef vax
	dsubinfo[0] = ubasetup(&dsb[0], 1);
# endif vax

	/*
	 * If there is a buffer offset
	 * then adjust b_addr temporarily.
	 * NOTE: this fudging around probably could
	 * be done unconditionally since f_boff
	 * is zeroed in dsopen so if the user
	 * hasn't specified a boff then this
	 * shouldn't affect b_addr.
	 */
	if (ds.d_state & DSBOFF) {
		oaddr = dsb[0].b_un.b_addr;
		dsb[0].b_un.b_addr += dsf.f_boff;
		dsb[0].b_un.b_addr = oaddr;
/* debug  printf("dsstart: size=%d\n", size(dsf.f_ccnt) - dsf.f_boff); */
		dsr->dmablr = ~ ((size(dsf.f_ccnt) - dsf.f_boff) >> 1);
		dsf.f_ccnt -= size(dsf.f_ccnt) - dsf.f_boff;
# ifdef pdp11
		dsr->dmasax = dsb[0].b_xmem & 03;
		dsr->dmasar = dsb[0].b_un.b_addr;
# endif pdp11
# ifdef vax
		dsr->dmasax = (dsubinfo[0] >> 16) & 03;
		dsr->dmasar = dsubinfo[0];
# endif vax
	}
	else {
/* debug  printf("dsstart: size=%d\n", size(dsf.f_ccnt)); */
		dsr->dmablr = ~ (size(dsf.f_ccnt) >> 1);
		dsf.f_ccnt -= size(dsf.f_ccnt);
# ifdef pdp11
		dsr->dmasax = dsb[0].b_xmem & 03;
		dsr->dmasar = dsb[0].b_un.b_addr;
# endif pdp11
# ifdef vax
		dsr->dmasax = (dsubinfo[0] >> 16) & 03;
		dsr->dmasar = dsubinfo[0];
# endif vax
	}

# ifdef vax
	dsubinfo[1] = ubasetup(&dsb[1], 1);
# endif vax

/* debug  printf("dsstart: size=%d\n", size(dsf.f_ccnt - size(dsf.f_ccnt))); */
	dsr->dmablr = ~ (size(dsf.f_ccnt - size(dsf.f_ccnt)) >> 1);
	dsf.f_ccnt -= size(dsf.f_ccnt - size(dsf.f_ccnt));
# ifdef pdp11
	dsr->dmasax = dsb[1].b_xmem & 03;
	dsr->dmasar = dsb[1].b_un.b_addr;
# endif pdp11
# ifdef vax
	dsr->dmasax = (dsubinfo[1] >> 16) & 03;
	dsr->dmasar = dsubinfo[1];
# endif vax

	dsr->asccsr |= RUN;

	/*
	 * Wait for interrupt routine to signal
	 * end of conversions.
	 */
	while (ds.d_state & DSBSY)
		sleep((caddr_t) &ds, DSPRI);

	/*
	 * Wait for disk.
	 * Free uba.
	 */
out:	for (i = 0; i < NDSB; i++) {
		/*
		 * If we got a dsreset then don't
		 * wait for the buffers to get done
		 * since the converters were put to
		 * bed and we'll never finish them up.
		 */
		if (ds.d_errs & DSRST) {
			int waittime;
			for (waittime = 0; waittime < 5; waittime++)
				sleep((caddr_t) &lbolt, DSPRI);
		}
		else if (ds.d_errs & DSDISK) {
			if ((dsb[i].b_flags & (B_DONE|B_ERROR)) == 0) {
				int waittime;
				for (waittime = 0; waittime < 5; waittime++)
					sleep((caddr_t) &lbolt, DSPRI);
				printf("ds: diskio[%d] wait error\n", i);
			}
		}
		else {
			while ((dsb[i].b_flags & B_DONE) == 0)
				sleep((caddr_t) &dsb[i], DSPRI);
		}
# ifdef vax
		if (dsubinfo[i] != 0) {
			ubafree(dsubinfo[i]);
			dsubinfo[i] = 0;
		}
# endif vax
	}

# ifdef vax
	vsunlock(base, count, B_READ);	/* somebody was reading */
# endif vax

	u.u_procp->p_flag &= ~ SULOCK;
	u.u_procp->p_flag |= flag;

# ifdef notdef	/* this was certainly stupid */
	u.u_count = count - dsf.f_ccnt;
	u.u_offset += count - dsf.f_ccnt;
# endif notdef

	if (ds.d_errs)
bad:		u.u_error = EIO;
}

/*
 * This is where the real work is done. We copy any device registers that
 * we will be looking at to decrease the amount of traffic on the dsc 200
 * bus. Also you can't do a "read-modify-write" (I think that this is called
 * a DATIP followed by a DATO in DEC manuals) on any of the dsc registers
 * while the converters are running.
 *
 * Note that you must write something to the dma11 clear status register
 * or you'll never get any more interrupts.
 *
 * Ds_wticks gets cleared on each interrupt. Dswatch() increments
 * ds_wticks and if ds_wticks gets over a threshold then the
 * addacs are assumed to have hung and we do a reset.
 */
dsintr() {
	register struct dsregs *dsr;
	register struct ds *dsp;
	register struct dsf *dsfp;
	register struct buf *bp;
	register int bufno;
	register int i;

	dsr = DSADDR;
	dsp = &ds;
	dsfp = &dsf;
	bufno = dsp->d_bufno % NDSB;
	bp = &dsb[bufno];

	dsp->d_asccsr = dsr->asccsr;
	dsp->d_dmacsr = dsr->dmacsr;

	/*
	 * Check to see if any disking
	 * needs to be done.
	 */
	if (dsfp->f_dcnt > 0) {
		bp->b_blkno = block(dsfp->f_dcnt, dsfp->f_todo);
		if ((bp->b_flags & B_DONE) == 0) {
			dsp->d_errs |= DSDISK;
			goto err;
		}
		if ((dsp->d_state & DSNDSK) == 0) {
			bp->b_flags &= ~ (B_DONE | B_ERROR);
			diskio(bp);
		}
		else
			bp->b_flags |= B_DONE;
		dsfp->f_dcnt -= dsfp->f_bsize;
	}

	/*
	 * Check to see if converting
	 * is finished.
	 */
	if ((dsfp->f_icnt -= dsfp->f_bsize) <= 0)
		goto out;

	if ((dsp->d_dmacsr & (ERR | BSY)) != BSY) {
		dsp->d_errs |= DSCERR;
err:		printf("dsc error; asc=%o, dsc=%o\n", dsp->d_asccsr, dsp->d_dmacsr);
out:		dsr->asccsr = 0;			/* clear run */
		dsr->dmacsr = 0;			/* turn off dma11 */
		dsp->d_state &= ~ DSBSY;
		wakeup((caddr_t) &ds);
		return;
	}

	/*
	 * Attempt to catch disk errors.
	 */
	for (i = 0; i < NDSB; i++)
		if (dsb[i].b_flags & B_ERROR) {
			dsp->d_errs |= DSDISK;
			goto err;
		}

	/*
	 * reinitialize dma11
	 */
	dsr->dmacls = 0;

	/*
	 * Load dma11 registers.
	 */
	if (dsfp->f_ccnt > 0) {
/* debug  printf("dsintr: size=%d\n", size(dsfp->f_ccnt)); */
		dsr->dmablr = ~ (size(dsfp->f_ccnt) >> 1);
# ifdef pdp11
		dsr->dmasax = dsb[bufno].b_xmem & 03;
		dsr->dmasar = dsb[bufno].b_un.b_addr;
# endif pdp11
# ifdef vax
		dsr->dmasax = (dsubinfo[bufno] >> 16) & 03;
		dsr->dmasar = dsubinfo[bufno];
# endif vax
	}

	/*
	 * Clear wticks.
	 */
	ds_wticks = 0;

	/*
	 * Update byte count.
	 * Update i/o count.
	 */
	dsfp->f_ccnt -= dsfp->f_bsize;
	dsp->d_bufno++;

/* # define ZEROBUF */
# ifdef ZEROBUF
	/*
	 * Last buffer is full of
	 * zeroes.
	 */
	if (dsfp->f_ccnt <= dsfp->f_bsize) {
		bufno = dsp->d_bufno % NDSB;
		dsbclr(dsb[bufno].b_un.b_addr, dsb[bufno].b_bcount);
	}
# endif ZEROBUF
}

dsbclr(base, bcnt)
caddr_t base;
unsigned int bcnt;
{
# ifdef vax
	/* debugging stuff */
	if (! kernacc(base, bcnt, B_READ)) {
		printf("can't dsbclr\n");
		return;
	}
	{ asm("	movc5	$0,(sp),$0,8(ap),*4(ap)"); }
# else vax
	register int i;
	register caddr_t cp;

	cp = base;
	i = bcnt;
	do
		*cp++ = 0;
	while (--i)
# endif vax
}

dsread() {
	dsstart(B_READ);	/* writes on disk */
}

dswrite() {
	dsstart(B_WRITE);	/* reads from disk */
}

dsioctl(dev, cmd, addr, flag)
caddr_t addr;
{
	register struct dsregs *dsr;
	struct ds_seq ds_seq;
	struct ds_err ds_err;
	struct ds_fs ds_fs;

	dsr = DSADDR;

	switch (cmd) {
		case DSSEQ:	/* set sequence */
			if (copyin(addr, &ds_seq, sizeof(struct ds_seq))) {
				u.u_error = EFAULT;
				return;
			}
			if ((ds_seq.reg > 15) || (ds_seq.reg < 0)) {
				u.u_error = EINVAL;
				return;
			}
			dsseq((int) ds_seq.reg, (int) ds_seq.conv, (int) ds_seq.dirt);
			break;
		case DSLAST:	/* mark last sequence register */
			if (copyin(addr, &ds_seq, sizeof(struct ds_seq))) {
				u.u_error = EFAULT;
				return;
			}
			if ((ds_seq.reg > 15) || (ds_seq.reg < 0)) {
				u.u_error = EINVAL;
				return;
			}
			dsr->ascseq[ds_seq.reg] |= bit(7);
			ds.d_args |= A_SEQ;
			break;
		case DSBNO:
			if (copyin(addr, &ds_fs, sizeof(struct ds_fs))) {
				u.u_error = EFAULT;
				return;
			}
			dsf.f_bno = ds_fs.bnosiz;
			ds.d_args |= A_BNO;
			break;
# ifdef notdef
		case DSTODO:
			if (copyin(addr, &ds_fs, sizeof(struct ds_fs))) {
				u.u_error = EFAULT;
				return;
			}
			dsf.f_todo = ds_fs.bnosiz;
			ds.d_args |= A_TODO;
			break;
# endif notdef
		case DSCOUNT:
			if (copyin(addr, &ds_fs, sizeof(struct ds_fs))) {
				u.u_error = EFAULT;
				return;
			}
			dsf.f_todo = dsf.f_dcnt = dsf.f_ccnt = dsf.f_icnt = ds_fs.bnosiz;
			ds.d_args |= A_CNT;
			break;
		case DSRATE:	/* set sample rate */
			if (copyin(addr, &ds_seq, sizeof(struct ds_seq))) {
				u.u_error = EFAULT;
				return;
			}
			dsr->ascsrt = ds_seq.dirt;
			break;
		case DS20KHZ:
			dsr->asccsr &= ~ HZMSK;	/* turn off all filter bits */
			dsr->asccsr |= HZ20;	/* set 20kHz filter */
			break;
		case DS10KHZ:
			dsr->asccsr &= ~ HZMSK;	/* turn off all filter bits */
			dsr->asccsr |= HZ10;	/* set 10kHz filter */
			break;
		case DS5KHZ:
			dsr->asccsr &= ~ HZMSK;	/* turn off all filter bits */
			dsr->asccsr |= HZ05;	/* set 5kHz filter */
			break;
		case DSBYPAS:
			dsr->asccsr &= ~ HZMSK;	/* turn off all filter bits */
			dsr->asccsr |= BYPASS;	/* set bypass */
			break;
		case DSNODSK:
			ds.d_state |= DSNDSK;
			break;
		case DSERRS:
			ds_err.dma_csr = ds.d_dmacsr;
			ds_err.asc_csr = ds.d_asccsr;
			ds_err.errors = ds.d_errs;
			if (copyout(&ds_err, addr, sizeof(struct ds_err))) {
				u.u_error = EFAULT;
				return;
			}
			break;
		case DSBOFF:
			if (copyin(addr, &ds_fs, sizeof(struct ds_fs))) {
				u.u_error = EFAULT;
				return;
			}
			if (ds_fs.bnosiz & 01) {
				u.u_error = EINVAL;
				return;
			}
			dsf.f_boff = ds_fs.bnosiz;
			break;
		default:
			u.u_error = ENOTTY;
			break;
	}
}

/*
 * Set up asc sequence registers.
 * Dir == 0 means d/a transfer,
 * dir == 1 means a/d transfer.
 */
dsseq(reg, conv, dir) {
	register struct dsregs *dsr;
	register int i, j;

	dsr = DSADDR;

	if (dir &= 01)			/* make sure low bit only is set */
		dir <<= 6;		/* bit 6 */

	dsr->ascseq[reg] = conv | dir;
}

/*
 * Ds_wticks gets cleared on each interrupt.Dswatch()
 * increments ds_wticks and if ds_wticks gets over
 * a threshold then the addacs are assumed to have
 * hung and we do a reset.
 */
dswatch() {
	if ((ds.d_state & DSOPEN) == 0) {
		ds_wticks = 0;
		return;
	}

	timeout(dswatch, (caddr_t) 0, HZ);

	ds_wticks++;

	if (ds_wticks >= 20) {
		ds_wticks = 0;
		printf("LOST INTERRUPT RESET");
		dsreset();
		printf("\n");
	}
}

# ifdef vax
/*
 * Zero uba vector.
 * Shut off converters.
 * Set error bit.
 */
dsreset() {
	register struct dsregs *dsr;
	register short dummy;
	register int i;

	dsr = DSADDR;

	printf(" ds");
	for (i = 0; i < NDSB; i++) {
		if (dsubinfo[i] != 0) {
			printf("<%d>", (dsubinfo[i] >> 28) & 0xf);
			ubafree(dsubinfo[i]);
			dsubinfo[i] = 0;
		}
	}

	if (dsr->dmacsr & OFL)
		printf(" (asc offline)");
	else {
		dsr->asccsr = 0;
		dsr->ascrst = 0;
	}

	dsr->dmablr = 0;
	dsr->dmasax = 0;
	dsr->dmasar = 0;
	dsr->dmacsr = 0;
	dsr->dmawc = 0;
	dsr->dmaacx = 0;
	dsr->dmaac = 0;
	dsr->dmadr = 0;
	dsr->dmaiva = 0;
	dsr->dmaclr = 0;
	dsr->dmacls = 0;
	dummy = dsr->dmasar;			/* clear sar flag */

	/*
	 * Put converters in monitor mode.
	 */
 	dsseq(0, ADBASE+0, AD);
 	dsseq(1, ADBASE+1, AD);
	dsr->ascseq[1] |= bit(7);		/* last seq */
	dsr->asccsr = RUN | MON;

	/*
	 * Abnormal termination.
	 */
	ds.d_errs |= DSCERR | DSRST;
	ds.d_state &= ~ DSBSY;
	wakeup((caddr_t) &ds);
}
# endif vax
# endif
