/* ds.c	@(#)ds.c	1.2	2/23/88	*/
/* To test on a machine without converters you can define FAKE 
   and then the 4.2 stuff will act like it is
   working and exit after preloading the play buffers. 
   It also needs a change to autoconf.c to call the attach routine as
   there are no DACS available */

#include "ds.h"
# if NDS > 0

/* define UNIXFILES for 4.2 capabilities. With UNIXFILES one can use
   ONEBDP for using only one BDP instead of one per buffer. It appears that
   if the DACS and a UDA50 are on the same bus that ONEBDP will not be
   able to get one large contiguous space. 

   Define CARLFILES to include table (and usage of it) of devices for UCSD 
   Csound type files.
   
   SYNC will turn on the wait for a signal from some event (in our case
   an interrupt on a special terminal line) to syncronize as much as
   possible playing with a tape recorder (generating a click). */


/*
 * DSC System 200 driver
 * via DSC dma11.
 * vax 4.2bsd version (can play both CARL and 4.2 files)
 * (4.2 files must have fragsize = blksize and be regular).
 * Problems:
 *	1). the number of BDP on the 750 and using more than 2 buffers
 * 	    Solutions would be to try direct DMA rather than BDP transfers, get
 * 	    uba resources on the fly, use only one BDP and doing the partition
 * 	    ourself (#define ONEBDP).
 *	    VAX 730 don't have this problem as they don't have
 * 	    buffered data paths and ubasetup turns it off anyway.
 *
 *	2). To play files with holes (no actual disk block allocated) you
 *	    need to be able to use a known disk block with zeros in it as
 *	    uba resourses are locked and from the interrupt routine you
 *	    can't use copyout() to user space as who knows who is the user.
 *	    A solution is to use the boot block which is always allocated
 *	    from mkfs, and can be cleared. It is of no use on a soundfile
 *	    file system. We expanded the file system size to 16K without
 *	    changing the location of the super block (8K into file system)
 *	    so there is a check to see (if there are empty blocks) if the
 *	    byte count for transfer is greater than BBSIZE. At this time we
 *	    are using the first data block available to a new file system
 *	    and locking it up with zeros via an altered mkfs.
 */


/*
 * !!! WARNING !!! WARNING !!! WARNING !!!
 *
 * since the disk driver strategy routine
 * is called from the converters' interrupt
 * level the processor level must not be set
 * to 0 in the disk driver. also where the
 * processor level is raised before fiddling
 * with the buffer list it must not be "raised"
 * to lower than the dma11's interrupt level.
 * change the code in the strategy routine to
 * look something like this:
 *
 *	# define spl5 spl6	(* converters run at level 6 *)
 *
 *	opl = spl5();		(* CHANGED *)
 *	dp = &xxutab[ui->ui_unit];
 *	disksort(dp, bp);
 *	if (dp->b_active == 0) {
 *		xxustart(ui);
 *		bp = &ui->ui_mi->um_tab;
 *		if ((bp->b_actf != NULL) && (bp->b_active == 0))
 *			xxstart(ui->ui_mi);
 *	}
 *	(void) splx(opl);	(* CHANGED *)
 *
 * you should also bracket the entire disk interrupt routine
 * with "opl = spl6() ... splx(opl)" and put an "splx(opl)"
 * prior to each return in it.
 *
 * the other alternative is to fix your dma11 so that it
 * interrupts at level 5 instead of level 6. dsc can give
 * you the information on how to do this. in either event
 * the strategy routine must be fixed as shown above.
 */

#include "../h/param.h"
#include "../h/systm.h"
#include "../h/mount.h"
#include "../h/dir.h"
#include "../h/user.h"
#include "../machine/pte.h"
#include "../h/map.h"
#include "../h/buf.h"
#include "../h/kernel.h"
#include "../vaxuba/ubavar.h"
#include "../h/conf.h"
#include "../h/proc.h"
#include "../h/uio.h"
#include "../h/file.h"
#include "../vaxuba/dsc.h"
#include "../vaxuba/dsreg.h"

#ifdef UNIXFILES
#include "../h/inode.h"
#include "../h/fs.h"
#include "../machine/cpu.h"
#include "../vaxuba/ubareg.h"
#endif




#ifdef CARLFILES
/*
 * THESE ARE SITE-SPECIFIC
 *
 * bulk storage devices
 *
 * these can be tape, disk, bubble
 * or whatever.
 */

# include "up.h"
# include "hp.h"
# include "tu.h"
# include "te.h"
# include "ra.h"

# if	NSC > 0
extern int upstrategy();
# endif	NSC

# if	NRA > 0
extern int udstrategy();
# endif	NRA

# if	NHP > 0
extern int hpstrategy();
# endif	NHP

# if	NHT > 0
extern int htstrategy();
# endif	NHT


struct bs {
	dev_t	d_cdev;		/* chrdev; major+minor */
	dev_t	d_bdev;		/* blkdev; major+minor */
	int	d_flags;	/* buffer flags */
	int	(*d_io)();	/* strategy routine */
} bs[] = {
# if	NUP > 0
	/* rup0c */		/* up0c */
	{ makedev(13, 2), 	makedev(2, 2), 	0, 	upstrategy },
	/* rup2f */		/* up2f */
	{ makedev(13, 21), 	makedev(2, 21), 	0, 	upstrategy },
# endif	NUP

# if	NRA > 0
	/* rra0f */		/* ra0f */
	{ makedev(9, 5), 	makedev(9, 5), 	0, 	udstrategy },
	/* rra1f */		/* ra1f */
	{ makedev(9, 13), 	makedev(9, 13), 	0, 	udstrategy },
	/* rra2f */		/* ra1f */
	{ makedev(9, 21), 	makedev(9, 21), 	0, 	udstrategy },
# endif	NRA

# if	NHT > 0
	/* rmt8 */		/* mt8 */
	{ makedev(5, 8), 	makedev(1, 8), 	B_TAPE, 	htstrategy },
# endif	NHT
};
# define NBS		(sizeof(bs) / sizeof(bs[0]))
# endif CARLFILES


/*
 * THESE ARE SITE-SPECIFIC
 *
 * starting base for the
 * d/a and a/d converters,
 * for setting up sequence ram.
 */
# define ADBASE		040
# define DABASE		050


# define WATCHDOG	40


/*
 * reset device
 */
# define RESETDEV	bit(4)

# define DSUNIT(dev)		(minor(dev) & ~ RESETDEV)
# define DSRESETUNIT(dev)	(minor(dev) & RESETDEV)

extern daddr_t	dsblock();

# define size(sf, cnt) ((cnt > (sf)->f_bsize) ? (sf)->f_bsize : cnt)

# define dsseq(ui, reg, conv, dir) \
((struct dsdevice *) (ui)->ui_addr)->ascseq[reg] = conv | ((dir & 01) << 6)

/*
 * ds flags
 */
# define DS_CLOSED	0
# define DS_OPEN	bit(0)
# define DS_BSY		bit(1)
# define DS_NDSK	bit(2)
# define DS_MON		bit(3)
# define DS_BRD		bit(4)

# ifdef SYNC
/* Multi-track tape synchronisation */
/* Hang on open from terminal line off a trigger */

#define WAITROUTINE dmintr

extern int WAITROUTINE();

# define DS_SYNC	bit(5)
# define DS_SYNCTIMEOUT	bit(6)

/* Default timeout for SYNC */
# define DEFSYNCTIME	WATCHDOG - 10

/* Max timeout */
# define SYNCTIMEMAX	(60*6)

# endif SYNC


# ifdef UNIXFILES
# define DS_42BSD	bit(7)
# define DS_LOADING	bit(8)
# define DS_RESET	bit(9)
# define CHAIN_REGISTERS 2

int LOADING_TIMEOUT = 10;
int	dscdebug = 0;
int	ndsb;

#endif UNIXFILES


/*
 * params to driver
 */
# define A_TODO		bit(0)
# define A_BNO		bit(1)
# define A_CNT		bit(2)
# define A_SEQ		bit(3)
# define A_DEV		bit(4)
# define A_NBLKS	bit(5)
# define A_BLIST	bit(6)

# define MINARG 	(A_BNO | A_CNT | A_SEQ | A_DEV)
# define DSPRI		(PZERO-1)

/* size of the buffer holding the block list */
# define LISTSIZE	MAXBSIZE

/* number of disk addresses per buffer */
# define NDADDRS	(LISTSIZE / sizeof(daddr_t))

/*
 * relevant information
 * about the dma and asc
 */
struct ds_softc {
	int		c_dmacsr;	/* copy of dma csr on error */
	int		c_asccsr;	/* copy of asc csr on error */
	int		c_flags;	/* internal flags */
	int		c_errs;		/* errors, returned via ioctl */
	int		c_bufno;	/* dsubinfo/buffer */
	int		c_uid;		/* user id */
	int		c_args;		/* args received from user */
	int		c_wticks;	/* watch dog */
	int		c_ubinfo[NDSB];	/* uba info */
	int		c_bubinfo;	/* uba info, 1st buffer, once only */
	int		c_nblist;	/* length of block list */
	int		c_blkno;	/* current block in blist */
	int		c_mode;		/* mode opened for */
#ifdef SYNC
	int		c_timeoutval;	/* Number of seconds to timeout sync */
#endif
	struct buf	c_dsb[NDSB];	/* bs buffers */
	struct buf	c_blist;	/* block list */
	struct buf	*c_curb;	/* current buffer in block list */
} ds_softc[NDS];

/*
 * relevant information
 * about the disking and
 * other truck
 */
struct ds_softf {
	daddr_t	f_bno;			/* starting block */
	off_t	f_todo;			/* amnt. of file to convert */
	off_t	f_ccnt;			/* amnt. of data not converted */
	off_t	f_dcnt;			/* amnt. of file not diskio'd */
	off_t	f_icnt;			/* amnt. of file not seen by dsintr */
	off_t	f_blr[NDSB];		/* amnt. last converted	*/
	int	f_bsize;		/* size of each buffer */
	int	f_boff;			/* offset into buffer of first i/o */
	int	f_dev;			/* bs device to use */
#ifdef UNIXFILES
	int	(*f_strat)();		/* Stratagey routine for samples */
	int	f_fd;			/* 4.2 file descriptor */
	off_t	f_extra;		/* silence at EOF */
#endif
} ds_softf[NDS];

int dsprobe(), dsattach(), dsintr();

struct uba_device *dsdinfo[NDS];

u_short dsstd[] = {
	0164400, 0
};

struct uba_driver dsdriver = {
	dsprobe, 0, dsattach, 0, dsstd, "ds", dsdinfo
};

/* ARGSUSED */
dsopen(dev, mode)
	dev_t dev;
{
	register struct uba_device *ui;
	register struct ds_softc *sc;
	register int unit;

	if ((unit = DSUNIT(dev)) >= NDS)
		goto bad;
#ifndef FAKE
	if ((ui = dsdinfo[unit]) == NULL)
		goto bad;

	if (ui->ui_alive == 0)
		goto bad;
	sc = &ds_softc[ui->ui_unit];
#else FAKE
	sc = &ds_softc[0];
#endif

	/*
	 * if this is the reset device
	 * then just do a reset and return.
	 */
	if (DSRESETUNIT(dev)) {
		/*
		 * if the converters are in use then
		 * only the current user or root can
		 * do a reset.
		 */
		if (sc->c_flags & DS_OPEN) {
			if ((sc->c_uid != u.u_ruid) && (u.u_uid != 0)) {
				return(ENXIO);
			}
		}

		if (dsinit(unit)) {
			uprintf("ds%d: asc offline\n", ui->ui_unit);
			return(EIO);
		}

		return(0);
	}

	/*
	 * only one person can use it
	 * at a time
	 */
	if (sc->c_flags & DS_OPEN)
bad:		return(ENXIO);

	sc->c_flags &= DS_OPEN; /* Make sure the device is ours now */
	sc->c_mode = mode;


	/*
	 * initialize
	 */
	if (dsinit(unit)) {
		uprintf("ds%d: asc offline\n", ui->ui_unit);
		return(EIO);
	}

	sc->c_flags = DS_OPEN; /* Will reset SYNC that may have been left */

	sc->c_uid = u.u_ruid;

	return(0);
}

/* ARGSUSED */
dsclose(dev, flag) {
	register int unit;

	unit = DSUNIT(dev);
	ds_softc[unit].c_flags = DS_CLOSED; /* Will turn off all flags */
	(void) dsinit(unit);

	return(0);
}

dsinit(unit) {
	register struct dsdevice *dsaddr;
	register struct uba_device *ui;
	register struct ds_softc *sc;
	register struct ds_softf *sf;
	register short dummy;
	register int offl;
	int flts;
	int odd;
#ifdef UNIXFILES
	int loadtime;
#endif

	if (unit >= NDS)
		return(1);
#ifdef FAKE
	ui = dsdinfo[0];
#else
	if ((ui = dsdinfo[unit]) == NULL)
		return(1);
#endif FAKE

#ifndef FAKE
	dsaddr = (struct dsdevice *) ui->ui_addr;

	if (dsaddr->dmacsr & DMA_OFL)
		offl = 1;
	else {
		flts = dsaddr->asccsr & ASC_HZMSK;	/* save filters */
		dsaddr->asccsr = flts;
		dsaddr->ascrst = 0;
		offl = 0;
	}
#endif
#ifdef FAKE

	offl = 0;
	sc = &ds_softc[0];
	sf = &ds_softf[0];
#else
	sc = &ds_softc[ui->ui_unit];
	sf = &ds_softf[ui->ui_unit];
#endif

#ifdef UNIXFILES
	loadtime = 0;
	while(sc->c_flags & DS_LOADING) { /* Sensitive loading */
		if(loadtime++ < LOADING_TIMEOUT) 
			sleep((char *) &lbolt,DSPRI);
		else {
			sc->c_flags &= ~ DS_LOADING;
			printf("ds%d: loading timeout, flags = %O\n",
				ui->ui_unit,sc->c_flags);
		}
	}
#endif
	odd = 0;

#ifndef FAKE

	/*
	 * flush out last remaining buffer
	 */
#ifdef UNIXFILES
	if((sc->c_flags & DS_RESET) == 0) /* only if not reseting unibus */
#endif
	if ((sc->c_mode & FREAD)	&&
	   (sf->f_dcnt > 0)		&&
	   (sc->c_flags & DS_BSY)	&&
	   (! (sc->c_flags & DS_NDSK)))	{
		register struct buf *bp;

#ifdef UNIXFILES
	if(sc->c_flags & DS_42BSD)  /* Adjust acording to uio */
		ndsb = NDSB;
	else
		ndsb = 2;
		
		bp = &sc->c_dsb[sc->c_bufno % ndsb];
#else UNIXFILES
		bp = &sc->c_dsb[sc->c_bufno % NDSB];
#endif UNIXFILES

		/* BEGIN DEBUG */
		if (sf->f_bsize == 0) {
			uprintf("dsinit: zero bsize\n");
			sc->c_errs |= EDS_CERR;
			goto out;
		}
		/* END DEBUG */

		/* sometimes dmawc is odd? */
		bp->b_bcount = ~ dsaddr->dmawc;
		if ((bp->b_bcount % sizeof(short)) != 0)
			odd = bp->b_bcount;
		bp->b_bcount -= bp->b_bcount % sizeof(short);
		bp->b_blkno = dsblock(sf, sc);
		bp->b_flags &= ~ (B_DONE | B_ERROR);
#ifdef UNIXFILES
		if(sf->f_strat != 0)
			(*sf->f_strat)(bp);
		else
			printf("dsc: Zero strategy routine\n");
#else UNIXFILES
		(*bs[sf->f_dev].d_io)(bp);
#endif UNIXFILES

		sf->f_dcnt -= sf->f_bsize;

		out:;
	}
#endif FAKE

	dsfreeall(&sc->c_blist);

#ifndef FAKE
	dsaddr->dmablr = 0;
	dsaddr->dmasax = 0;
	dsaddr->dmasar = 0;
	dsaddr->dmacsr = 0;
	dsaddr->dmawc = 0;
	dsaddr->dmaacx = 0;
	dsaddr->dmaac = 0;
	dsaddr->dmadr = 0;
	dsaddr->dmaiva = 0;
	dsaddr->dmaclr = 0;
	dsaddr->dmacls = 0;
	dummy = dsaddr->dmasar;			/* clears sar flag */

# endif FAKE

	/* reset ds_softc */
	sc->c_dmacsr = 0;
	sc->c_asccsr = 0;

#ifdef SYNC
	/* Return to dsstart sleep() for running process needs SYNC flags */
	sc->c_flags &= (DS_SYNC | DS_SYNCTIMEOUT); 
#endif
	sc->c_errs = 0;
	sc->c_bufno = 0;
	sc->c_args = 0;
	sc->c_blkno = 0;
	sc->c_mode = 0;

#ifdef SYNC
	sc->c_timeoutval = 0;
#endif

	/* reset ds_softf */
#ifdef FAKE
	sf = &ds_softf[0];
#else
	sf = &ds_softf[ui->ui_unit];
#endif
	sf->f_bno = 0;
	sf->f_todo = 0;
	sf->f_ccnt = 0;
	sf->f_dcnt = 0;
	sf->f_icnt = 0;
	sf->f_bsize = 0;
	sf->f_boff = 0;
	sf->f_dev = 0;

#ifdef UNIXFILES
	sf->f_strat = 0;
	sf->f_fd = -1;
	sf->f_extra = 0;
#endif

	/*
	 * terminate current run
	 */
	sc->c_flags = 0;

#ifndef FAKE
	wakeup((caddr_t) &ds_softc[ui->ui_unit]);
#endif
#ifdef SYNC
	if(sc->c_flags & DS_SYNC)
		wakeup((caddr_t) WAITROUTINE);
#endif
	if (sc->c_flags & DS_BSY)
		sc->c_errs |= EDS_RST;

	if (odd)
		printf("ds%d: dsinit: odd dmawc (%d)\n", ui->ui_unit, odd);

	return(offl);
}

/*
 * f_bsize is the size of the buffers used for the i/o. the buffers are
 * obtained by taking the user buffer and splitting it into NDSB buffers.
 * For CARL files the user buffer should be a multiple of the track 
 * size of the disk. For 4.2 it should be a multiple of the file system
 * block size.
 *
 * using uio.uio_len, uio.uio_base, and f_bsize each buffer header is set up.
 * the converters only need the base address of the buffer and the word
 * count. the disk needs these in addition to the block number. if we
 * are doing d/a conversions then we start the disk up to fill up the
 * buffers. For 4.2 files the f_offset (file pointer) is used so one can 'seek'
 * in to the file.
 *
 * after everything is ready to go we turn on the RUN bit and let 'em rip.
 */
dsstart(unit, rw, uio)
	struct uio *uio;
{
	register struct dsdevice *dsaddr;
	register struct uba_device *ui;
	register struct ds_softc *sc;
	register struct ds_softf *sf;
	register struct iovec *iov;
	register struct buf *bp;
	int dummy;
	int bits;
	int blr;
	int opl;
	int i ;

	int cbufsize;

	sc = &ds_softc[unit];
	sf = &ds_softf[unit];

#ifndef FAKE
	ui = dsdinfo[unit];
	dsaddr = (struct dsdevice *) ui->ui_addr;
#endif FAKE
	

	iov = uio->uio_iov;

	/*
	 * make sure we have all
	 * necessary info.
	 */

#ifdef UNIXFILES
	sc->c_flags |= DS_LOADING;	/* Don't let dsinit get to us */
	if(sc->c_flags & DS_42BSD) {  /* 4.2 file setup. */
		int ret ;

		ndsb = NDSB;
		/* Disk operation is opposite of DAC/ADC operation */
		if(ret = build42list(sc,sf,uio,(rw == B_READ ? B_WRITE : B_READ))) {
			sc->c_flags &= ~ DS_LOADING;
			return(ret);
		}
		if(dscdebug)
			uprintf("4.2 soundfile. start blk = %d, byte off = %d\n",
				dsblock(sf,sc),sf->f_boff);
	}
	else {
		ndsb = 2; /* Stantard for CARL files */
		sf->f_bsize = (iov->iov_len / ndsb);
	}
#endif UNIXFILES

	if ((sc->c_args & MINARG) != MINARG) {
		sc->c_errs |= EDS_ARGS;
		goto bad;
	}

	/*
	 * either the converters will be writing
	 * to memory or the disk will be.
	 */
	cbufsize = (sc->c_flags & DS_42BSD ? 
		sf->f_bsize * ndsb : (int) iov->iov_len);

	if (useracc(iov->iov_base, cbufsize, B_WRITE) == NULL) {

#ifdef UNIXFILES
		sc->c_flags &= ~ DS_LOADING;
#endif
		sc->c_errs |= EDS_ACC;
		return(EFAULT);
	}

	/*
	 * check for misaligned buffer.
	 * 
	 * the "% DEV_BSIZE" check isn't correct. it doesn't hurt
	 * though. it probably should be deleted.
	 */
	if (((cbufsize % DEV_BSIZE) != 0) || (cbufsize & 01)) {
		sc->c_errs |= EDS_MOD;
		goto bad;
	}

	/*
	 * check for tiny buffer
	 */
	if (sf->f_bsize < DEV_BSIZE) {
		sc->c_errs |= EDS_SIZE;
		goto bad;
	}

	/*
	 * check for unreasonable buffer
	 * offset for first buffer
	 */
	if (sf->f_boff >= sf->f_bsize) {
		sc->c_errs |= EDS_ARGS;
		goto bad;
	}
	
#ifdef UNIXFILES
	sf->f_dcnt = sf->f_ccnt + sf->f_extra + sf->f_boff;
#else UNIXFILES
	sf->f_dcnt = sf->f_ccnt + sf->f_boff;
#endif UNIXFILES

	u.u_procp->p_flag |= SPHYSIO;

	sc->c_flags |= DS_BSY;

	
	vslock(iov->iov_base, (int) cbufsize);

	/*
	 * point each c_dsb somewhere into
	 * the user's buffer, set up each c_dsb.
	 * the buffer's rw flag is set to the
	 * inverse of our operation since it
	 * applies to the bulk storage device
	 * i/o to be done.
	 */
#ifdef UNIXFILES
	for (i = 0; i < ndsb; i++) {
#else UNIXFILES
	for (i = 0; i < NDSB; i++) {
#endif UNIXFILES
		bp = &sc->c_dsb[i];
		bp->b_un.b_addr = iov->iov_base + (i * sf->f_bsize);
		bp->b_error = 0;
		bp->b_proc = u.u_procp;
		bp->b_bcount = sf->f_bsize;
#ifdef UNIXFILES
		if(sc->c_flags & DS_42BSD) {
			bp->b_flags =  B_BUSY | bdevsw[major(sf->f_dev)].d_flags | 
				B_PHYS | ((rw == B_READ) ? B_WRITE : B_READ);
			bp->b_dev = sf->f_dev;
#ifdef ONEBDP
			if(i == 0) { /* Grab one big piece */
				bp->b_bcount = sf->f_bsize * ndsb;

#ifdef FAKE
				sc->c_ubinfo[i] = ubasetup(0,
#else FAKE
				sc->c_ubinfo[i] = ubasetup(ui->ui_ubanum, 
#endif FAKE
					&sc->c_dsb[i], UBA_NEEDBDP|UBA_CANTWAIT);
				/* reset individual */
				bp->b_bcount = sf->f_bsize;
			}
			sc->c_ubinfo[i] = sc->c_ubinfo[0] + i * sf->f_bsize;
#else ONEBDP
#if VAX750
	/* Disk needs one. */
# define MAX750BDP (NBDP750 - 1)
	/* Turn off buffered Data Path request as it will hang. */

			sc->c_ubinfo[i] = 
#ifdef FAKE
				ubasetup(0, &sc->c_dsb[i],
#else FAKE
				ubasetup(ui->ui_ubanum, &sc->c_dsb[i],
#endif FAKE
				((cpu == VAX_750 && ndsb > MAX750BDP) 
					? UBA_CANTWAIT :  UBA_NEEDBDP|UBA_CANTWAIT));
#else VAX750
			sc->c_ubinfo[i] = 
#ifdef FAKE
				ubasetup(0, &sc->c_dsb[i], UBA_CANTWAIT|UBA_NEEDBDP);
#else FAKE
				ubasetup(ui->ui_ubanum, &sc->c_dsb[i], UBA_CANTWAIT|UBA_NEEDBDP);
#endif FAKE
#endif VAX750
#endif ONEBDP

			if(dscdebug)
				if(sc->c_ubinfo[i] == 0) {
					printf("Zero return from ubasetup (buf %d)\n",i);
					goto out;
				}
		}
# ifdef CARLFILES
		else {
			bp->b_flags = bs[sf->f_dev].d_flags | B_PHYS | 
				((rw == B_READ) ? B_WRITE : B_READ);
			bp->b_flags |= B_BUSY;
			bp->b_dev = bs[sf->f_dev].d_cdev;
		}
# endif CARLFILES
#else UNIXFILES
		bp->b_flags = bs[sf->f_dev].d_flags | B_PHYS | 
			((rw == B_READ) ? B_WRITE : B_READ);
		bp->b_flags |= B_BUSY;
		bp->b_dev = bs[sf->f_dev].d_cdev;
#endif UNIXFILES
		if (rw == B_WRITE) {
			if ((sc->c_flags & DS_NDSK) == 0) {
				bp->b_blkno = dsblock(sf, sc);
				sc->c_blkno++;
				bp->b_flags &= ~ (B_DONE | B_ERROR);

#ifdef UNIXFILES
if(dscdebug)
	uprintf("dsstart: blk=%ld, count=%d, ubinfo = %X\n", 
			bp->b_blkno, bp->b_bcount,sc->c_ubinfo[i]);
				if(sf->f_strat != 0)
					(*sf->f_strat)(bp);
				else {
					printf("Zero disk strat routine for DACS\n");
					sc->c_errs |= EDS_DISK;
					goto out;
				}
#else UNIXFILES
				(*bs[sf->f_dev].d_io)(bp);
#endif UNIXFILES

				/* iodone will wake us up */
				while ((bp->b_flags & B_DONE) == 0)
					sleep((caddr_t) bp, DSPRI);

				/* oops */
				if (bp->b_flags & B_ERROR) {
					sc->c_errs |= EDS_DISK;
					goto out;
				}
			}
			else
				bp->b_flags |= B_DONE;

			sf->f_dcnt -= sf->f_bsize;
		}
		else 
			bp->b_flags |= B_DONE;


	}

#ifndef FAKE

	opl = spl6();
#ifdef UNIXFILES
	sc->c_flags &= ~ DS_LOADING;
#endif UNIXFILES
	/*
	 * if reading then going to
	 * memory so set W2M
	 */
	if (rw == B_WRITE)
		dsaddr->asccsr |= ASC_PLAY;
	else {
		dsaddr->asccsr |= ASC_RECORD;
		dsaddr->dmacsr |= DMA_W2M;
	}

	dsaddr->dmacsr |= DMA_CHN | DMA_SFL;
	dummy = dsaddr->dmasar;
	dsaddr->asccsr &= ~ ASC_MON;
	dummy = dsaddr->ascrst;
 	dsaddr->asccsr |= ASC_IE;
	dsaddr->dmacls = 0;
	dsaddr->dmacsr |= DMA_IE;
	dsaddr->dmawc = -1;

	if ((blr = dssize(sf)) <= 0) {
		sc->c_errs |= EDS_ARGS;
		splx(opl);
		goto out;
	}
	sf->f_blr[0] = blr;

#ifdef UNIXFILES
	if((sc->c_flags & DS_42BSD) == 0) 
#ifdef FAKE
		sc->c_ubinfo[0] = ubasetup(0, &sc->c_dsb[0], UBA_NEEDBDP);
#else FAKE
		sc->c_ubinfo[0] = ubasetup(ui->ui_ubanum, &sc->c_dsb[0], UBA_NEEDBDP);
#endif FAKE
#else UNIXFILES
#ifdef FAKE
		sc->c_ubinfo[0] = ubasetup(0, &sc->c_dsb[0], UBA_NEEDBDP);
#else FAKE
		sc->c_ubinfo[0] = ubasetup(ui->ui_ubanum, &sc->c_dsb[0], UBA_NEEDBDP);
#endif FAKE
#endif UNIXFILES

	sc->c_bubinfo = sc->c_ubinfo[0] + sf->f_boff;
	dsaddr->dmablr = ~ (blr >> 1);
	dsaddr->dmasax = (sc->c_bubinfo >> 16) & 03;
	dsaddr->dmasar = sc->c_bubinfo;

#else FAKE
#ifdef UNIXFILES
	sc->c_flags &= ~ DS_LOADING;
#endif
#endif FAKE
	sf->f_ccnt -= blr;

	/*
	 * all done with boff; nail it.
	 */
	 sf->f_boff = 0;

# ifdef debug
	if (sf->f_ccnt <= 0)
		uprintf("dsstart: dsb0: f_ccnt=%d\n", sf->f_ccnt);
# endif debug

#ifdef UNIXFILES
	if((sc->c_flags & DS_42BSD) == 0) 
#ifdef FAKE
		sc->c_ubinfo[1] = ubasetup(0, &sc->c_dsb[1], UBA_NEEDBDP);
#else FAKE
		sc->c_ubinfo[1] = ubasetup(ui->ui_ubanum, &sc->c_dsb[1], UBA_NEEDBDP);
#endif FAKE
#else UNIXFILES
#ifdef FAKE
	sc->c_ubinfo[1] = ubasetup(0,&sc->c_dsb[1], UBA_NEEDBDP);
#else FAKE
	sc->c_ubinfo[1] = ubasetup(ui->ui_ubanum, &sc->c_dsb[1], UBA_NEEDBDP);
#endif FAKE
#endif UNIXFILES

	/*
	 * set monitor mode or
	 * broadcast mode.
	 */
	bits = 0;
	if (sc->c_flags & DS_MON)
		bits |= ASC_MON;
	if (sc->c_flags & DS_BRD)
		bits |= ASC_BRD;

#ifdef SYNC

	if(sc->c_flags & DS_SYNC)
	{
		sleep((caddr_t) WAITROUTINE, DSPRI);
		sc->c_flags &= ~ DS_SYNC; /* Turn it off */
	}
	/* If in sync wait and someone killed the process don't play file */
	/* If we had a sync wait timeout then don't play file */
	if(((sc->c_flags & DS_BSY) == 0) || (sc->c_flags & DS_SYNCTIMEOUT)) {
		if(sc->c_flags & DS_SYNCTIMEOUT) {
			sc->c_flags &= ~ DS_SYNCTIMEOUT; /* Turn it off */
			sc->c_flags &= ~ DS_BSY;
			sc->c_errs |= EDS_TIMEOUT;
		}
		splx(opl);
		goto out;
	}
#endif SYNC

#ifndef FAKE
	/*
	 * as they say in california,
	 * ``go for it''
	 */
	dsaddr->asccsr |= ASC_RUN | bits;


	/*
	 * if the file is so small and only uses the first
	 * buffer we would load dmablr with 0 but we can't
	 * do that because we get errors (data late, external
	 * interrupt), so we fudge and give it f_bsize. dsintr
	 * will shut down the converters before it tries to play
	 * the 2nd (bogus) block.
	 */
	if ((blr = dssize(sf)) <= 0) {
# ifdef debug
		uprintf("dsstart: dsb1: blr=%d\n", blr);
# endif debug
		blr = sf->f_bsize;
	}
	sf->f_blr[1] = blr;

	dsaddr->dmablr = ~ (blr >> 1);
	dsaddr->dmasax = (sc->c_ubinfo[1] >> 16) & 03;
	dsaddr->dmasar = sc->c_ubinfo[1];
	sf->f_ccnt -= blr;
# ifdef debug
	if (sf->f_ccnt <= 0)
		uprintf("dsstart: dsb1: f_ccnt=%d\n", sf->f_ccnt);
# endif debug
	splx(opl);
#ifdef SYNC
	if(sc->c_timeoutval) /* sync used */
		uprintf("\n");	/* show him we have started */
#endif

	/*
	 * wait for interrupt routine to signal
	 * end of conversions. Non interruptable by signal
	 */
	while (sc->c_flags & DS_BSY)
		sleep((caddr_t) &ds_softc[ui->ui_unit], DSPRI);

		

	/*
	 * wait for disk i/o to complete.
	 * release unibus resources.
	 */
#else FAKE
#ifdef SYNC
	if(sc->c_timeoutval)
		uprintf("\n");	/* show him we are going */
#endif SYNC
#endif FAKE

#ifdef UNIXFILES
out:	for (i = 0; i < ndsb; i++) {
#else UNIXFILES
out:	for (i = 0; i < NDSB; i++) {
#endif UNIXFILES
		if ((sc->c_dsb[i].b_flags & B_DONE) == 0) {
			int waittime;
			for (waittime = 0; waittime < 7; waittime++) {
				sleep((caddr_t) &lbolt, DSPRI);
				if (sc->c_dsb[i].b_flags & B_DONE)
					goto done;
			}
			printf("ds%d: dsb%d: wait timeout\n", ui->ui_unit, i);
		}

done:		if (sc->c_ubinfo[i] != 0) {

#ifdef UNIXFILES
#ifdef ONEBDP
			if(i == 0) {
				if(dscdebug)
					uprintf("ubarelse (onebdp) : %X\n",
						sc->c_ubinfo[i]);
#ifdef FAKE
				ubarelse(0, &sc->c_ubinfo[0]);
#else FAKE
				ubarelse(ui->ui_ubanum, &sc->c_ubinfo[0]);
#endif FAKE
			}
#else ONEBDP
			if(dscdebug)
				uprintf("ubarelse: %X\n",sc->c_ubinfo[i]);
#ifdef FAKE
			ubarelse(ui->ui_ubanum, &sc->c_ubinfo[i]);
#else FAKE
			ubarelse(ui->ui_ubanum, &sc->c_ubinfo[i]);
#endif FAKE
#endif ONEBDP
#else UNIXFILES
#ifdef FAKE
			ubarelse(0, &sc->c_ubinfo[i]);
#else FAKE
			ubarelse(ui->ui_ubanum, &sc->c_ubinfo[i]);
#endif FAKE
#endif UNIXFILES
			sc->c_ubinfo[i] = 0;
		}
	}

	/*
	 * either the disk or the
	 * converters were writing
	 * to memory, thus the B_READ.
	 */
	vsunlock(iov->iov_base, (int) cbufsize, B_READ);

	u.u_procp->p_flag &= ~ SPHYSIO;

	if (sc->c_errs) {
bad:		
#ifdef UNIXFILES
	sc->c_flags &= ~ DS_LOADING;
#endif
		return(EIO);
	}

	return(0);
}

/*
 * this is where the real work is done. we copy any device registers that
 * we will be looking at to decrease the amount of traffic on the ds 200
 * bus. also you can't do a "read-modify-write" (I think that this is called
 * a DATIP followed by a DATO in DEC manuals) on any of the ds registers
 * while the converters are running.
 *
 * note that you must write something to the dma11 clear status register
 * or you'll never get any more interrupts.
 *
 * c_wticks gets cleared on each interrupt. dswatch() increments
 * c_wticks and if c_wticks gets over a threshold then the
 * addacs are assumed to have hung and we do a reset.
 */
dsintr(dev)
	dev_t dev;
{
	register struct dsdevice *dsaddr;
	register struct ds_softc *sc;
	register struct ds_softf *sf;
	register struct buf *bp;
	register int bufno;
	register int i;
	int unit;
	int blr;

#ifndef FAKE
	unit = DSUNIT(dev);
	sc = &ds_softc[unit];
	sf = &ds_softf[unit];

	/*
	 * reset c_wticks
	 */
	sc->c_wticks = 0;

	dsaddr = (struct dsdevice *) dsdinfo[unit]->ui_addr;

	sc->c_asccsr = dsaddr->asccsr;
	sc->c_dmacsr = dsaddr->dmacsr;

#ifdef UNIXFILES
 	bufno = (sc->c_bufno + 1) % ndsb;
#else UNIXFILES
 	bufno = (sc->c_bufno + 1) % NDSB;
#endif UNIXFILES
 	bp = &sc->c_dsb[bufno];
 	if (
#ifdef UNIXFILES
	dscdebug && 
#endif UNIXFILES
	(bp->b_flags & B_DONE) == 0)
 	    printf("dsintr: dac has chained to buffer which is not B_DONE.\n");
 
	/*
	 * get current buffer
	 */
#ifdef UNIXFILES
	bufno = sc->c_bufno % ndsb;
#else
	bufno = sc->c_bufno % NDSB;
#endif UNIXFILES
	bp = &sc->c_dsb[bufno];

	/*
	 * check to see if any disking
	 * needs to be done
	 */
	if (sf->f_dcnt > 0) {
		if ((bp->b_flags & B_DONE) == 0) {
			sc->c_errs |= EDS_DISK;
			goto out;
		}

		if ((sc->c_flags & DS_NDSK) == 0) {
			if (sf->f_bsize == 0) {
				uprintf("dsintr: zero bsize\n");
				sc->c_errs |= EDS_CERR;
				goto out;
			}

			bp->b_blkno = dsblock(sf, sc);
			sc->c_blkno++;

			bp->b_flags &= ~ (B_DONE | B_ERROR);
#ifdef UNIXFILES
			(*sf->f_strat)(bp);
#else UNIXFILES
			(*bs[sf->f_dev].d_io)(bp);
#endif UNIXFILES
		}
		else
			bp->b_flags |= B_DONE;

		sf->f_dcnt -= sf->f_bsize;
	}

	/*
	 * check to see if converting is finished.
	 * we need to decrement by the actual amount
	 * converted, not just the buffer size.
	 */
	if ((sf->f_icnt -= sf->f_blr[bufno]) <= 0)
		goto out;

	/*
	 * check for converter error.
	 * all errors and normal termination
	 * come here.
	 */
	if ((sc->c_dmacsr & (DMA_ERR | DMA_BSY)) != DMA_BSY) {
		int flts;
		sc->c_errs |= EDS_CERR;
		printf("ds%d error: asccsr=%b, dmacsr=%b\n", unit, 
			sc->c_asccsr, ASC_BITS, sc->c_dmacsr, DMA_BITS);

out:		sc->c_flags &= ~ DS_BSY;
		flts = dsaddr->asccsr & ASC_HZMSK;	/* save filters */
		dsaddr->asccsr = flts;			/* clears run */
		dsaddr->dmacsr = 0;			/* turn off dma11 */
		wakeup((caddr_t) &ds_softc[unit]);
		return(0);
	}

	/*
	 * reinitialize dma11
	 */
	dsaddr->dmacls = 0;

	/*
	 * load dma11 registers
	 */
#ifdef UNIXFILES
	/* Set up for next play buffer */
 	bufno = (sc->c_bufno + CHAIN_REGISTERS) % ndsb;
#endif UNIXFILES
	blr = dssize(sf);
	if (sf->f_ccnt > 0) {
		dsaddr->dmablr = ~ (blr >> 1);
		dsaddr->dmasax = (sc->c_ubinfo[bufno] >> 16) & 03;
		dsaddr->dmasar = sc->c_ubinfo[bufno];
	}
	sf->f_blr[bufno] = blr;

	/*
	 * try to catch disk errors
	 */
#ifdef UNIXFILES
	for (i = 0; i < ndsb; i++) {
#else UNIXFILES
	for (i = 0; i < NDSB; i++) {
#endif UNIXFILES
		if (sc->c_dsb[i].b_flags & B_ERROR) {
			sc->c_errs |= EDS_DISK;
			goto out;
		}
	}

	/*
	 * update converted byte count.
	 * update buffers converted count.
	 */
	sf->f_ccnt -= blr;
	sc->c_bufno++;

# ifdef ZEROBUF
	/*
	 * last buffer is full of
	 * zeroes
	 */
	if (sf->f_ccnt <= sf->f_bsize) {
#ifdef UNIXFILES
		bufno = sc->c_bufno % ndsb;
#else UNIXFILES
		bufno = sc->c_bufno % NDSB;
#endif UNIXFILES
		dsbclr(sc->c_dsb[bufno].b_un.b_addr, sc->c_dsb[bufno].b_bcount);
	}
# endif ZEROBUF

#endif FAKE
	return(0);
}

#ifdef notdef
/*
 * last time i checked, this
 * didn't work
 */
dsbclr(base, bcnt)
	caddr_t base;
	unsigned int bcnt;
{
	/* debugging stuff */
	if (! kernacc(base, bcnt, B_READ)) {
		uprintf("can't dsbclr\n");
		return;
	}
	/* end debugging stuff */

	{ asm("	movc5	$0,(sp),$0,8(ap),*4(ap)"); }
}
#endif notdef

/*
 * a/d conversion
 */
dsread(dev, uio)
	dev_t dev;
	struct uio *uio;
{
	int unit;

	unit = DSUNIT(dev);
	return(dsstart(unit, B_READ, uio));	/* writes on disk */
}

/*
 * d/a conversion
 */
dswrite(dev, uio)
	dev_t dev;
	struct uio *uio;
{
	int unit;

	unit = DSUNIT(dev);
	return(dsstart(unit, B_WRITE, uio));	/* reads from disk */
}

/* ARGSUSED */
dsioctl(dev, cmd, addr, flag)
	dev_t dev;
	caddr_t addr;
{
	register struct dsdevice *dsaddr;
	register struct ds_softc *sc;
	register struct ds_softf *sf;
	register struct ds_seq *dq;
	register struct ds_err *de;
	register struct ds_fs *df;
# ifdef MAJMIN
	register struct bs *bsp;
# endif MAJMIN
	struct uba_device *ui;
	struct ds_seq ds_seq;
	struct ds_err ds_err;
	struct ds_fs ds_fs;
	int unit;
	int i;

	unit = DSUNIT(dev);

	sc = &ds_softc[unit];
	sf = &ds_softf[unit];
	ui = dsdinfo[unit];
#ifndef FAKE
	dsaddr = (struct dsdevice *) ui->ui_addr;
#endif

	switch (cmd) {
#ifdef SYNC
		/* SYNC mode with tape click from dh line (dh.c) */
		case DSSYNC:
			sc->c_flags |= DS_SYNC;
			break;

		case DSSYNCTIMEVAL:
			df = &ds_fs;
			bcopy(addr, (caddr_t) df, sizeof(struct ds_fs));
			if (df->bnosiz < 0 || df->bnosiz > SYNCTIMEMAX)
				return(EINVAL);
			sc->c_timeoutval = df->bnosiz;
			break;
#endif
		/* put in broadcast mode */
		case DSBRD:
			if (! (sc->c_args & A_SEQ)) {
				sc->c_errs |= EDS_ARGS;
				return(EINVAL);
			}

			sc->c_flags |= DS_BRD;
			break;

		/* put in monitor mode */
		case DSMON:
			if (! (sc->c_args & A_SEQ)) {
				sc->c_errs |= EDS_ARGS;
				return(EINVAL);
			}

			sc->c_flags |= DS_MON;

			break;

		/* set sequence */
		case DSSEQ:
			dq = &ds_seq;
			bcopy(addr, (caddr_t) dq, sizeof(struct ds_seq));
			if ((dq->reg > 15) || (dq->reg < 0))
				return(EINVAL);

#ifndef FAKE
			dsseq(ui, (int) dq->reg, (int) dq->conv, (int) dq->dirt);
#endif
			break;

		/* mark last sequence register */
		case DSLAST:
			dq = &ds_seq;
			bcopy(addr, (caddr_t) dq, sizeof(struct ds_seq));
			if ((dq->reg > 15) || (dq->reg < 0))
				return(EINVAL);

#ifndef FAKE
			dsaddr->ascseq[dq->reg] |= bit(7);
#endif
			sc->c_args |= A_SEQ;
			break;

		/* starting block number */
		case DSBNO:
			df = &ds_fs;
			bcopy(addr, (caddr_t) df, sizeof(struct ds_fs));
			if (df->bnosiz < 0)
				return(EINVAL);

			sf->f_bno = df->bnosiz;
			sc->c_args |= A_BNO;
			break;

		/* no. of bytes to convert */
		case DSCOUNT:
			df = &ds_fs;
			bcopy(addr, (caddr_t) df, sizeof(struct ds_fs));

			/*
			 * DSCOUNT is in bytes so
			 * must be modulo sizeof(short)
			 */
			if ((df->bnosiz & 01) || (df->bnosiz <= 0))
				return(EINVAL);

			sf->f_todo = sf->f_ccnt = sf->f_icnt = df->bnosiz;
			sc->c_args |= A_CNT;
			break;

# ifdef CARLFILES
		/* bs device to use */
		case DSDEV:
			dq = &ds_seq;
			bcopy(addr, (caddr_t) dq, sizeof(struct ds_seq));

# ifdef	MAJMIN
			for (bsp = &bs[0]; bsp < &bs[NBS]; bsp++) {
				if (dq->dirt == bsp->d_cdev)
					break;
			}

			if (bsp == &bs[NBS])
				return(EINVAL);

			if (bsdevchk(dq->dirt) == -1)
				return(EBUSY);
# else	MAJMIN
			if ((dq->dirt >= NBS) || (dq->dirt < 0))
				return(EINVAL);

			if (bsdevchk(bs[dq->dirt].d_bdev) == -1)
				return(EBUSY);
# endif	MAJMIN

			sf->f_dev = dq->dirt;
			sc->c_args |= A_DEV;
#ifdef UNIXFILES
#ifdef MAJMIN
			sf->f_strat = bsp.d_io;
#else
			sf->f_strat = bs[dq->dirt].d_io;
#endif MAJMIN
#endif UNIXFILES
			break;
# endif CARLFILES

		/* set sample rate */
		case DSRATE:
			dq = &ds_seq;
			bcopy(addr, (caddr_t) dq, sizeof(struct ds_seq));
#ifndef FAKE
			dsaddr->ascsrt = dq->dirt;
#endif
			break;

		case DS20KHZ:
#ifndef FAKE
			dsaddr->asccsr &= ~ ASC_HZMSK;
			dsaddr->asccsr |= ASC_HZ20;	/* set 20kHz filter */
#endif
			break;

		case DS10KHZ:
#ifndef FAKE
			dsaddr->asccsr &= ~ ASC_HZMSK;
			dsaddr->asccsr |= ASC_HZ10;	/* set 10kHz filter */
#endif
			break;

		case DS5KHZ:
#ifndef FAKE
			dsaddr->asccsr &= ~ ASC_HZMSK;
			dsaddr->asccsr |= ASC_HZ05;	/* set 5kHz filter */
#endif FAKE
			break;

		case DSBYPAS:
#ifndef FAKE
			dsaddr->asccsr &= ~ ASC_HZMSK;
			dsaddr->asccsr |= ASC_BYPASS;	/* set bypass */
#endif FAKE
			break;

		/* no bs device i/o */
		case DSNODSK:
			sc->c_flags |= DS_NDSK;
			sc->c_args |= A_DEV | A_BNO;
			break;

		/* fetch errors */
		case DSERRS:
			de = &ds_err;
#ifndef FAKE
			de->dma_csr = sc->c_dmacsr;
			de->asc_csr = sc->c_asccsr;
#endif FAKE
			de->errors = sc->c_errs;
			bcopy((caddr_t) de, addr, sizeof(struct ds_err));
			break;

		/* byte offset into 1st buffer */
		case DSBOFF:
			df = &ds_fs;
			bcopy(addr, (caddr_t) df, sizeof(struct ds_fs));

			/*
			 * DSBOFF is in bytes so
			 * must be modulo sizeof(short)
			 */
			if ((df->bnosiz & 01) || (df->bnosiz < 0))
				return(EINVAL);

			sf->f_boff = df->bnosiz;
			break;

		/* how many samples actually converted */
		case DSDONE:
			df = &ds_fs;
			df->bnosiz = sf->f_todo - sf->f_icnt;
#ifndef FAKE
			df->bnosiz += ~ dsaddr->dmawc;
#endif
			bcopy((caddr_t) df, addr, sizeof(struct ds_fs));
			break;

#ifdef UNIXFILES
		case DS42BSD:

			/* Use a 4.2 file */
			df = &ds_fs;
			bcopy(addr, (caddr_t) df, sizeof(struct ds_fs));

			/* file descriptor */
			if (df->bnosiz < 0)
				return(EINVAL);

			sf->f_fd = df->bnosiz;

			sc->c_flags |= DS_42BSD;
			break;
#endif UNIXFILES

		case DSNBLKS:
			df = &ds_fs;
			bcopy(addr, (caddr_t) df, sizeof(struct ds_fs));

			if (df->bnosiz < 1)
				return(EINVAL);

			sc->c_nblist = df->bnosiz;
			sc->c_args |= A_NBLKS;
			break;

		case DSBLKS:
			if ((sc->c_args & A_NBLKS) == 0) {
				sc->c_errs |= EDS_ARGS;
				return(EINVAL);
			}

			for (i = sc->c_nblist * sizeof(daddr_t); i > 0; i -= LISTSIZE) {
				struct buf	*bp;
				unsigned int	amt;

				bp = geteblk(LISTSIZE);
				clrbuf(bp);

				amt = i > LISTSIZE ? LISTSIZE : i;

				if (copyin(*(caddr_t *)addr, bp->b_un.b_addr, amt)) {
					dsfreeall(&sc->c_blist);
					return(EFAULT);
				}

				*(caddr_t *)addr += amt;

				dslink(&sc->c_blist, bp);
			}
			sc->c_curb = sc->c_blist.av_forw;

			sc->c_args |= A_BNO | A_BLIST;
			break;

		default:
			return(ENOTTY);
			break;
	}

	return(0);
}

/*
 * link a buffer onto the buffer list
 * of blocks
 */
dslink(dp, bp)
	register struct buf	*dp, *bp;
{
	dp->av_back->av_forw = bp;
	bp->av_back = dp->av_back;
	dp->av_back = bp;
	bp->av_forw = dp;
}

/*
 * return all of the buffers to
 * the system
 */
dsfreeall(dp)
	register struct buf	*dp;
{
	register struct buf	*bp;

	for (bp = dp->av_forw; bp != dp; bp = dp->av_forw) {
		bp->av_back->av_forw = bp->av_forw;
		bp->av_forw->av_back = bp->av_back;
		brelse(bp);
	}
}

dssize(sf)
	register struct ds_softf	*sf;
{
#ifdef UNIXFILES
	int tmp;
#endif UNIXFILES
	/*
	 * normal case.
	 * f_boff is zeroed in the dsstart routine
	 * so its affect is only seen the first time.
	 */
	if (sf->f_ccnt >= sf->f_bsize)
		return(sf->f_bsize - sf->f_boff);

	/*
	 * convoluted code follows.
	 *
	 * this code handles the situation where we are playing
	 * something that is less than a track (f_bsize) but 
	 * there are three situations that have to be handled:
	 *   1. the section to be played lies entirely within
	 *	a track with unplayed space both in front of it
	 *	and in back of it.
	 *   2. the section to be played lies entirely within
	 *	a track but has unplayed space only in front of it.
	 *	(a special case of 1).
	 *   3. the section to be played extends across a track
	 *	boundary.
	 *
	 * f_todo was set to the number of bytes to be converted;
	 * it never changes.
	 */
	if (sf->f_todo == sf->f_ccnt) {
		if (sf->f_ccnt <= (sf->f_bsize - sf->f_boff))
			return(sf->f_ccnt);			/* 1 & 2 */
		else
			return(sf->f_bsize - sf->f_boff);	/* 3 */
	}

	/*
	 * this is executed for the last track
	 */
#ifdef UNIXFILES
	/* Is there added silence? */
	if(sf->f_extra > 0) {
		tmp = sf->f_ccnt;
		sf->f_ccnt += sf->f_extra;
		sf->f_extra = 0;
		return(tmp);
	}
#endif UNIXFILES
	return(sf->f_ccnt);
}

daddr_t
dsblock(sf, sc)
	register struct ds_softf	*sf;
	register struct ds_softc	*sc;
{
	daddr_t				blkno;
	daddr_t				*daddrs;

	if (sc->c_args & A_BLIST) {
		/* circular block list */
		blkno = sc->c_blkno % sc->c_nblist;

		/*
		 * if we're at the end of this buffer then
		 * move on to the next one.
		 */
		if ((blkno % NDADDRS) == 0) {
			/* special case for first time only */
			if (sc->c_bufno != 0)
				sc->c_curb = sc->c_curb->av_forw;
		}

		/*
		 * convert the buffer data from an array of
		 * bytes to an array of longs, then index
		 * into that to get the block.
		 */
		daddrs = (daddr_t *) sc->c_curb->b_un.b_addr;
		blkno = daddrs[blkno % NDADDRS];
	}
	else
		blkno = ((sf->f_todo - sf->f_dcnt) / 512) + sf->f_bno;

	return(blkno);
}

#ifdef DSUINIT
/*
 * user level entry to dsinit. check that
 * unit is valid and call dsinit. this is
 * used by user programs via the dsinit
 * system call to reset the dacs.
 */
dsuinit() {
	register struct uba_device *ui;
	register struct ds_softc *sc;
	register int unit;
	register struct a {
		int dsunit;
	} *uap;

	uap = (struct a *) u.u_ap;

	if ((unit = uap->dsunit) >= NDS)
		return(ENXIO);

#ifndef FAKE
	if ((ui = dsdinfo[unit]) == NULL)
		return(ENXIO);
	sc = &ds_softc[ui->ui_unit];
#else
	sc = &ds_softc[0];
#endif


	/*
	 * possible abnormal termination;
	 * only the current user or root can
	 * terminate an active run
	 */
	if (sc->c_flags & DS_OPEN) {
		if (sc->c_uid != u.u_ruid) {
			if (u.u_uid != 0)
				return(ENXIO);
		}
	}

	if (dsinit(ui->ui_unit)) {
		uprintf("ds%d: asc offline\n", ui->ui_unit);
		return(EIO);
	}

	return(0);
}
#endif DSUINIT


#ifdef CARLFILES
/*
 * check to see if the indicated bulk storage
 * device is a mounted filesystem.
 */
bsdevchk(bsdev)
	register int bsdev;
{
	register struct mount *mp;

	for (mp = &mount[0]; mp < &mount[NMOUNT]; mp++) {
		if (bsdev == mp->m_dev) {
			return(EBUSY);
		}
	}
	return(0);
}
#endif CARLFILES

/*
 * all this just to generate
 * an interrupt; rather
 * involved isn't it?
 */
dsprobe(reg)
	caddr_t reg;
{
	register int br, cvec;		/* value-result */
	register struct dsdevice *dsaddr;
	int dummy;

# ifdef lint	
	br = 0; cvec = br; br = cvec;
# endif lint

#ifdef FAKE
	cvec = 700;
	dsattach(0);
	return(sizeof(struct dsdevice));
#else
	dsaddr = (struct dsdevice *) reg;

	dsaddr->dmacsr = 0;
	dsaddr->dmacls = 0;
	dsaddr->ascseq[0] = DABASE+0 | ((DA & 01) << 6);
	dsaddr->ascseq[0] |= bit(7);	/* last sequence register */
	dsaddr->ascsrt = 0100;
	dsaddr->dmacsr = DMA_IE;
	dsaddr->asccsr = ASC_RUN | ASC_IE;

	DELAY(40000);

	/*
	 * now shut everything down.
	 * too bad we have to duplicate
	 * the code from dsinit but to
	 * call dsinit we need to give
	 * it a unit.
	 */
	if ((dsaddr->dmacsr & DMA_OFL) == 0) {
		dsaddr->asccsr = 0;
		dsaddr->ascrst = 0;
	}

	dsaddr->dmablr = 0;
	dsaddr->dmasax = 0;
	dsaddr->dmasar = 0;
	dsaddr->dmacsr = 0;
	dsaddr->dmawc = 0;
	dsaddr->dmaacx = 0;
	dsaddr->dmaac = 0;
	dsaddr->dmadr = 0;
	dsaddr->dmaiva = 0;
	dsaddr->dmaclr = 0;
	dsaddr->dmacls = 0;
	dummy = dsaddr->dmasar;			/* clears sar flag */

	return(sizeof(struct dsdevice));
#endif FAKE
}

dsattach(ui)
	struct uba_device *ui;
{
	extern int dswatch();
	static int dswstart = 0;
	register struct ds_softc *sc;

#ifdef FAKE
	printf("Fake DSC online\n");
#endif
	/*
	 * set up the blist linked list
	 * to the empty list. this must
	 * be done first because dsinit()
	 * calls dsfreeall().
	 */
#ifdef FAKE
	sc = &ds_softc[0];
#else
	sc = &ds_softc[ui->ui_unit];
#endif
	sc->c_blist.av_forw = &sc->c_blist;
	sc->c_blist.av_back = &sc->c_blist;

#ifdef FAKE
	(void) dsinit(0);
#else
	(void) dsinit(ui->ui_unit);
#endif

	/*
	 * start watchdog
	 */
	if (dswstart == 0) {
		timeout(dswatch, (caddr_t) 0, hz);
		dswstart++;
	}

	return(0);
}


/*
 * c_wticks gets cleared on each interrupt. dswatch()
 * increments c_wticks and if c_wticks gets over
 * a threshold then the addacs are assumed to have
 * hung and we do a reset.
 */
dswatch() {
	register struct uba_device *ui;
	register struct ds_softc *sc;
	register int ds;

	/* requeue us */
	timeout(dswatch, (caddr_t) 0, hz);

#ifdef FAKE 
	ds = 0;
	sc = &ds_softc[ds];
	if ((sc->c_flags & DS_BSY) == 0) {
		sc->c_wticks = 0;
	}
	sc->c_wticks++;
#else

	for (ds = 0; ds < NDS; ds++) {
		if ((ui = dsdinfo[ds]) == NULL)
			continue;
		if (ui->ui_alive == 0)
			continue;

		sc = &ds_softc[ds];
		if ((sc->c_flags & DS_BSY) == 0) {
			sc->c_wticks = 0;
			continue;
		}

		sc->c_wticks++;

#endif FAKE
#ifdef SYNC
	/* Catch SYNC timeout before lost interrupt is sensed. */
		if(sc->c_flags & DS_SYNC) {
			if (sc->c_wticks >= sc->c_timeoutval) {
				sc->c_flags |= DS_SYNCTIMEOUT;
				sc->c_wticks = 0; 
				wakeup((caddr_t) WAITROUTINE);
			}
		}
		else
#endif

		if (sc->c_wticks >= WATCHDOG) {
#ifdef UNIXFILES
			sc->c_flags |= DS_RESET;
#endif UNIXFILES
			sc->c_wticks = 0;
			printf("ds%d: lost interrupt\n", ds);
			ubareset(ui->ui_ubanum);
		}
#ifndef FAKE
	}
#endif FAKE
	return(0);
}

/*
 * zero uba vector.
 * shut off converters.
 * set error bit.
 */
dsreset(uban) {
	register struct uba_device *ui;
	register struct ds_softc *sc;
	register int ds;
	register int i;

#ifndef FAKE
	for (ds = 0; ds < NDS; ds++) {
		if ((ui = dsdinfo[ds]) == NULL)
			continue;
		if (ui->ui_alive == 0)
			continue;
		if (ui->ui_ubanum != uban)
			continue;

		printf(" ds%d", ds);

		sc = &ds_softc[ds];

#ifdef UNIXFILES
		sc->c_flags &= ~ DS_LOADING;
#endif UNIXFILES

		/*
		 * this used to be after the ubarelse's,
		 * seems like it should go before them
		 */
		sc->c_flags |= DS_RESET;
		if (dsinit(ds))
			printf("<dsinit error>");

		/*
		 * release unibus resources
		 */
		for (i = 0; i < NDSB; i++) {
			if (sc->c_ubinfo[i] != 0) {
				printf("<buf %d>", (sc->c_ubinfo[i] >> 28) & 0xf);
				sc->c_ubinfo[i] = 0;
			}
		}
	}
#endif FAKE

	return(0);
}

#ifdef UNIXFILES

/* Build a list of file system size block numbers just like the CARL cylinder
   list. All we need is the users 'fd' for the file. The list will be 
   in units of disk blocks (512 byte blocks) not file system blocks */

build42list(sc,sf,uio,rw)
	register struct ds_softc *sc;
	register struct ds_softf *sf;
	register struct uio *uio;
	int rw;
{
	int i,count,loc;
	register struct inode *ip;
	register struct file *fp;
	register struct fs *fs;
	daddr_t bmap();
	struct file *getinode();
	struct iovec *iov = uio->uio_iov;

	/* Fetch inode and file system */

	if((fp = getinode(sf->f_fd)) == NULL)
		return(EBADF);

	ip = (struct inode *) fp->f_data;
	fs = ip->i_fs;


	/* Validate file and file system for conversion */


	if((fp->f_offset & 01) || (iov->iov_len & 01)) /* bytes */
		return(EINVAL);
	if(fs->fs_fsize != fs->fs_bsize) /* frag != block */
		return(EINVAL);
	if((ip->i_mode & IFMT) != IFREG) /* Nothing but normal files allowed */
		return(EINVAL);


	/* Individual buffers (NDSB of them) are file system block size */
	sf->f_bsize = fs->fs_bsize;

	loc = lblkno(fs,fp->f_offset); /* Start at this block */
	sf->f_boff = fp->f_offset % sf->f_bsize; /* byte block offset */
	sc->c_nblist = lblkno(fs,blkroundup(fs,sf->f_boff + iov->iov_len));
	sf->f_todo = sf->f_icnt = iov->iov_len;

	sf->f_extra = fp->f_offset + iov->iov_len - ip->i_size;

	/* Is there requested silence? */
	if(sf->f_extra > 0)
		sf->f_ccnt = iov->iov_len - sf->f_extra;
	else {
		sf->f_extra = 0;
		sf->f_ccnt = sf->f_todo;
	}

	if(dscdebug)
		uprintf("file system blocks = %d\n",sc->c_nblist);

	if(sc->c_nblist <= 0)
		return(EBADF);

	count = 0;

	/* Flush buffers for young files (just like fsync()) */
	ilock(ip);

#define YOUNGTIME 60
	if(ip->i_mtime + YOUNGTIME > time.tv_sec) {
		if(dscdebug)
			uprintf("syncing file\n");
		syncip(ip);
	}
	if(rw == B_READ) /* Record could fill in holes with bmap() */
		iunlock(ip);

	for (i = sc->c_nblist * sizeof(daddr_t); i > 0; i -= LISTSIZE) {
		struct buf	*bp;
		unsigned int	amt;
		daddr_t	tmp;

		bp = geteblk(LISTSIZE);
		clrbuf(bp);

		amt = i > LISTSIZE ? LISTSIZE : i;

		for(; amt > 0; loc++,count++ ,amt -= sizeof(daddr_t)) {

			/* For writes blocks will get filled in */
			tmp = bmap(ip,loc,rw,fs->fs_bsize); /* File systm block */
		
/* #ifdef USEBBLOCK
			   if tmp < 0 then it is an empty block and we will
			   leave the list entry = 0 as this is the BOOT block
			   and will have been zeroed on soundfile systems.
   #else USEBBLOCK
			   if tmp < 0 then it is an empty block and we will
			   use a known location relative to the file system
			   that will have been zeroed on soundfile systems.
   #endif */

			if(tmp < 0) { 
#ifdef USEBBLOCK
				if(fs->fs_bsize > BBSIZE || rw == B_WRITE) {  
#else USEBBLOCK
				if(rw == B_WRITE) { 
#endif USEBBLOCK
if(dscdebug)
	uprintf("Illegal empty disk blocks\n");
					dsfreeall(&sc->c_blist);
#ifdef USEBBLOCK
					if(rw == B_WRITE)
#endif USERBBLOCK
						iunlock(ip);
					return(EFAULT);
				}
#ifndef USEBBLOCK
				/* Hack for now to known block */
				tmp = fs->fs_dblkno + 3; /* inode4 */
#endif
			}
#ifdef USEBBLOCK
			else
#endif
				*((daddr_t *) bp->b_un.b_addr + 
					(count % NDADDRS)) = fsbtodb(fs,tmp); 
		}
		dslink(&sc->c_blist, bp);
	}
	if(rw == B_WRITE)
		iunlock(ip);
	sc->c_curb = sc->c_blist.av_forw;
	sf->f_dev = ip->i_dev;
	sf->f_strat = bdevsw[major(ip->i_dev)].d_strategy;

	sc->c_args |= A_BLIST | A_BNO | A_DEV | A_CNT;
	return(0);
}
# endif UNIXFILES

# else
# ifdef DSUINIT
dsuinit() {
	return(0);
}
# endif DSUINIT
# endif
