/* %M%	%I%	(CARL)	%G%	%U% */

# include <stdio.h>
# include "connlist.h"
# include "aswdaemon.h"

/*
 * routines to manage connection lists
 */

static struct connlist	*clhead = NULL;

/*
 * add connection of "inum" to "onum"
 * to the connlist.
 */
addconn(onum, inum) {
	extern char *calloc();
	register struct connlist *tcl;	/* temp connlist */

	if (clhead == NULL) {
		if (clsetup() == RET_ERR)
			return(RET_ERR);
	}

	/*
	 * linear search for this connection
	 */
	for (tcl = clhead->cl_forw; tcl != clhead; tcl = tcl->cl_forw) {
		if ((inum == tcl->cl_input) && (onum == tcl->cl_output))
			return(RET_OK); /* already connected */
	}

	if ((tcl = (struct connlist *) calloc(1, sizeof(struct connlist))) == NULL) {
		err(ERR_CONNLIST, "addconn: out of memory");
		return(RET_ERR);
	}

	tcl->cl_input = inum;
	tcl->cl_output = onum;

	/* link onto end of list */
	tcl->cl_forw = clhead->cl_back->cl_forw;
	tcl->cl_back = clhead->cl_back;
	clhead->cl_back->cl_forw = tcl;
	clhead->cl_back = tcl;

	return(RET_OK);
}

/*
 * delete connection of "inum" to "onum"
 * from the connlist.
 */
delconn(onum, inum) {
	register struct connlist *tcl;
	register struct connlist *forw;
	register int dels;

	if (clhead == NULL) {
		info(INFO_CONNLIST, "delconn: empty connlist");
		return(RET_ERR);
	}

	/*
	 * search the connection list looking for a
	 * connection between what we want to disconnect.
	 * just in case, we search the entire list in
	 * case there are multiple connections.
	 */
	dels = 0;
	for (tcl = clhead->cl_forw; tcl != clhead; tcl = forw) {

		/* save forward link in case we toss tcl */
		forw = tcl->cl_forw;

		if ((tcl->cl_input == inum) && (tcl->cl_output == onum)) {
			/* unlink */
			tcl->cl_back->cl_forw = tcl->cl_forw;
			tcl->cl_forw->cl_back = tcl->cl_back;

			free((char *) tcl);

			dels++;
		}
	}

	if (dels == 0) {
		info(INFO_CONNLIST, "delconn: no such connection [%d,%d]", inum, onum);
		return(RET_ERR);
	}

	return(RET_OK);
}

/*
 * delete all connections
 * from the connlist.
 */
clrconn() {
	register struct connlist *tcl;

	if (clhead == NULL) {
		info(INFO_CONNLIST, "clrconn: empty connlist");
		return(RET_ERR);
	}

	while ((tcl = clhead->cl_forw) != clhead) {
		/* unlink */
		tcl->cl_back->cl_forw = tcl->cl_forw;
		tcl->cl_forw->cl_back = tcl->cl_back;

		free((char *) tcl);
	}

	return(RET_OK);
}

struct connlist *
isconn(num, type) {
	register struct connlist *tcl;

	/*
	 * a previous bug sometimes let -1 sneek in
	 * (see parse.y).
	 */
	if ((type != CL_INPUT) && (type != CL_OUTPUT)) {
		err(ERR_CONNLIST, "isconn: bad type (%d)", type);
		return(NULL);
	}

	if (clhead == NULL) {
		info(INFO_CONNLIST, "isconn: empty connlist");
		return(NULL);
	}

	/*
	 * linear search for this connection
	 */
	for (tcl = clhead->cl_forw; tcl != clhead; tcl = tcl->cl_forw) {
		if ((num == tcl->cl_input) && (type == CL_INPUT))
			return(tcl);
		if ((num == tcl->cl_output) && (type == CL_OUTPUT))
			return(tcl);
	}

	return(NULL);
}

/*
 * routine to initialize clhead
 */
clsetup() {
	if ((clhead = (struct connlist *) calloc(1, sizeof(struct connlist))) == NULL) {
		err(ERR_CONNLIST, "clsetup: out of memory (clhead)");
		return(RET_ERR);
	}

	clhead->cl_back = clhead;
	clhead->cl_forw = clhead;

	return(RET_OK);
}

# define PRBUFFS	2

/*
 * return a string representation of
 * the connlist. prconnlist returns the
 * address to a static array which is
 * overwritten on each call.
 */
char *
prconnlist() {
	register struct connlist *tcl;
	static char buf[BUFSIZ];
	char numbuf[16];
	int first;

	if (clhead == NULL)
		return("");

	strcpy(buf, "");

	first = 1;
	for (tcl = clhead->cl_forw; tcl != clhead; tcl = tcl->cl_forw) {
		sprintf(numbuf, "%d->%d", tcl->cl_output, tcl->cl_input);
		strcat(buf, first ? "" : ",");
		strcat(buf, numbuf);
		first = 0;
	}

	return(buf);
}

/*
 * clget gets the next connection from the connlist
 * or returns NULL if we've reached the end of the
 * list.
 */
struct connlist *
clget() {
	static struct connlist		*clgnext;

	if (clgnext == NULL) {
		if (clhead == NULL) {
			if (clsetup() == RET_ERR)
				return(NULL);
		}
		clgnext = clhead;
	}

	if ((clgnext = clgnext->cl_forw) == clhead)
		return(NULL);

	return(clgnext);
}
