#include <carl/Player.h>
#include "debug.h"

/* 
 * get_templ - get Player class structure from procedure address 
 * takes:
 * 	proc -	Player procedure address
 * returns:
 *	Player class structure
 */

Player *
get_templ(proc)
	register Player (*proc)();
{
	register struct hlist *hl;

	if ((hl = lookup((char *) proc, AT_CLASS)) == NULL)
		return(NULL);
	return(hl->hl_val.st_class);
}

/*
 * New_Player - instantiate new Player instance 
 * takes:
 * 	proc -	address of Player procedure
 * 	beg -	begin time of Player instance
 * 	dur -	duration of Player instance
 * 	stat -	run status of Player instance
 * returns:
 * 	address of new Player instance data structure
 * side effects:
 * 	Installs Player template on ST_INSTANCE symbol table.
 * 	Links it to list of Player instances.
 * 	Queues it to run at the stated begin time.
 * diagnostics:
 * 	New_player: no class template for <PLAYER>
 * 	New_Player: backward time reference=<TIME> for Player <PLAYER>
 * 	
 */

Player *
New_player(proc, beg, dur, stat)
	register Player (*proc)();
	register double beg, dur;
	register int stat;
{
	Extern Player *get_templ();
	register Player *c, *i;

	c = get_templ(proc);
	if (c == NULL) {
		fprintf(stderr,"New_player: no class template for %x\n", c);
		exit(1);
	}
	if ((i = mk_Player(&ilist)) == NULL)
		exit(1);
	(*c->iproc)(i, c);		/* setup instance structure */
	if (beg < Now)			/* force no backwards time refs. */
		beg = Now;
	i->beg = i->ptime = beg;
	i->dur = dur;
	i->runstat = stat;
	i->caller = Self;
	if (Self != NULL)
		Self->n_children++;
	if (beg < Now) {
		fprintf(stderr, 
		    "New_Player: backward time reference=%f for Player %s\n",
		    beg, i->name);
		exit(1);
	}
	set_wait(i, beg);
	if (_Pdebug & NEW_PLAYER)
		fprintf((FILE *)stderr,
		"New_player: %s, stat=%d, beg=%6.3f, dur=%6.3f, Now=%6.3f\n",
			i->name, i->runstat, i->beg, i->dur, Now);	
	return(i);
}

/*
 * Caller -	get pointer to instance data of parent
 * takes:
 * 	x -	pointer to our instance data
 * returns:
 * 	pointer to parent's instance data, else NULL if no parent.
 */

Player *
Caller(x)
	Player *x;
{
	return(x->caller);
}

/* 
 * Stop - set status of Player to P_STOPPED
 * takes:
 *	Player instance structure address
 * returns: 
 * 	previous status as follows
 * 		P_NOSTAT 	0
 * 		P_READY 	1
 * 		P_RUNNING 	2
 * 		P_STOPPED 	3
 * 		P_KILLED 	4
 * side effects:
 *	Sets Player_stat to P_NULL_PTR if no such Player,
 * 	and returns P_NOSTAT.
 */

Stop(x)
	register Player *x;
{
	register unsigned i;

	Player_stat = 0;
	if (x != NULL) {
		i = x->runstat;
		x->runstat = P_STOPPED;
		return(i);
	} else {
		Player_stat = P_NULL_PTR;
		return(P_NOSTAT);
	}
}

/* 
 * Player_status - return status of Player 
 * takes:
 *	Player instance structure pointer
 * returns:
 *	status, as follows
 * 		P_NOSTAT 	0
 * 		P_READY 	1
 * 		P_RUNNING 	2
 * 		P_STOPPED 	3
 * 		P_KILLED 	4
 * side effects:
 *	Sets Player_stat to P_NULL_PTR if no such Player,
 * 	and returns P_NOSTAT.
 */

Player_status(x)
	register Player *x;
{
	if (x != NULL)
		return(x->runstat);
	else
		return(P_NOSTAT);
}

/* 
 * Start - set Player instance run status to P_READY
 * takes:
 *	Player instance structure address
 * returns: 
 * 	previous status as follows
 * 		P_NOSTAT 	0
 * 		P_READY 	1
 * 		P_RUNNING 	2
 * 		P_STOPPED 	3
 * 		P_KILLED 	4
 * side effects:
 *	Sets Player_stat to P_NULL_PTR if no such Player,
 * 	and returns P_NOSTAT.
 */

Start(x)
	register Player *x;
{
	register unsigned i;

	Player_stat = 0;
	if (x != NULL) {
		i = x->runstat;
		x->runstat = P_READY;
		x->ptime = Now;
		set_wait(x, Now);
		return(i);
	} else {
		Player_stat = P_NULL_PTR;
		return(P_NOSTAT);
	}
}

/*
 * Resting - test and then clear Player rest status flag
 * takes:
 * 	x -	Player instance structure address
 * action:
 * 	clears status
 * returns:
 * 	old status
 */

Resting(x)
	register Player *x;
{
	register int i;
	if (x == NULL)
		return(-1);
	i = x->reststat;
	x->reststat = 0;
	return(i);
}

/* 
 * Rest - temporarily defeat Player instance output
 * takes:
 *	Player instance structure address
 * returns: 
 *	0 on success, -1 on failure
 * side effects:
 *	Sets Player_stat to P_NULL_PTR if no such Player.
 */

Rest(x)
	register Player *x;
{
	register unsigned i;

	if (x != NULL) {
		i = x->reststat;
		x->reststat = TRUE;
		Player_stat = 0;
		return(0);
	} else {
		Player_stat = P_NULL_PTR;
		return(-1);
	}
}

/* 
 * Unrest - reenable Player instance output
 * takes:
 *	Player instance structure address
 * returns: 
 *	0 on success, -1 on failure
 * side effects:
 *	sets Player_stat to P_NULL_PTR if no such Player.
 */

Unrest(x)
	register Player *x;
{
	register unsigned i;

	if (x != NULL) {
		i = x->reststat;
		x->reststat = FALSE;
		Player_stat = 0;
		return(i);
	} else {
		Player_stat = P_NULL_PTR;
		return(-1);
	}
}

/* 
 * Silent - permanently suppress Player instance output
 * takes:
 *	Player instance structure address
 * returns: 
 *	0 on success, -1 on failure
 * side effects:
 *	Sets Player_stat to P_NULL_PTR if no such Player.
 */

Silent(x)
	register Player *x;
{
	register unsigned i;

	if (x != NULL) {
		i = x->silent;
		x->silent = TRUE;
		Player_stat = 0;
		return(i);
	} else {
		Player_stat = P_NULL_PTR;
		return(-1);
	}
}

/* 
 * Unsilent - reenable Player instance output
 * takes:
 *	Player instance structure address
 * returns: 
 *	0 on success, -1 on failure
 * side effects:
 *	Sets Player_stat to P_NULL_PTR if no such Player.
 */

Unsilent(x)
	register Player *x;
{
	register unsigned i;

	if (x != NULL) {
		i = x->silent;
		x->silent = FALSE;
		Player_stat = 0;
		return(i);
	} else {
		Player_stat = P_NULL_PTR;
		return(-1);
	}
}

/*
 * Kill - terminate a process with extreme prejudice
 * Takes:
 * 	Player *x -	points to Player instance to be terminated
 * returns:
 * 	0 on success, -1 on errors
 * side effects:
 * 	If a parent process is waiting to be notified of the termination
 * 	of children, that parent is scheduled to run.  See Join()
 * 	for an explanation of conditions for this.
 *	The parent's status field, indicating the request to be run
 *	on child termination, will be cleared if there are no more
 *	children to wait on.
 * 	
 */

Kill(x)
	register Player *x;
{
	if (x == NULL)
		return(-1);
	x->runstat = P_KILLED;
	if (x->caller != NULL) {
		if (x->caller->child_wait == EACH_CHILD)
			set_wait(x->caller, Now);
		if (--x->caller->n_children == 0) {
			if (x->caller->child_wait == ALL_CHILDREN) {
				x->caller->ptime = Now;
				set_wait(x->caller, Now);
			}
			x->caller->child_wait = 0;
		}
	}
	garbage(x);
	return(0);
}

/*
 * Join -	cause the current instance to be awakened when children expire
 * Takes:
 * 	flag indicating condition to be rescheduled:
 * 		ALL_CHILDREN -	run only when all childern have been Kill()'ed
 * 		EACH_CHILD -	run when each child is Kill()'ed
 * returns:
 * 	nothing
 * side effects:
 * 	Sets a field on the current instance data structure to cause
 * 	the scheduler to do this.
 */

Join(flag)
	int flag;
{
	Self->child_wait = flag;
}
