/* allocsf.c	1.3	(CARL)	9/17/86	23:25:34 */
#include <stdio.h>
#include <carl/libsf.h>
extern char *malloc(), *calloc(), *realloc();

/*	
 * 	Storage list manager for cylinder based disk allocation system.
 *	Circular first-fit strategy, similar to malloc.
 *	Last link has flag == EOLIST, and is not used for storage.
 *	Works with contiguous, monotonically linked arena.
 *	Adjacent free blocks are coalesced during space search.
 *	Reads and writes interlocked file ALLOCNAME, which contains list.
 *	ALLOCNAME is defined in ./filesf.h
 */

/*
 * CAVEAT: (Or, in the immortal words of Gen. Haig, "I'd like to caveat
 * that...") The last entry in the in-core representation of the free list
 * is terminated by a block with flag == EOLIST.  That's because it is a
 * circular list where we need to know where it stops.   That block is NOT
 * written in the text file representation of the free list, ALLOCNAME,
 * because the end of the file is the end of the list.
 * 
 * What's more, the in-core representation of the cylinder list attached
 * to any sfd is NOT terminated by EOLIST, but by the end of the sfd->cp
 * list being == to NULL.  That's because it is a linear list and doesn't
 * need one.  Moreover, the cylinder list in an .sdf file does not have
 * an EOLIST element for the same reason.
 * 
 * So, when marching around the free list in memory, look to terminate on
 * EOLIST.  But when marching along an sfd->cp list, stop at a NULL node.
 */

/*
 * delink munches through a disk block list, sending it back to the bit
 * bucket.  If doing this to an sfd->cp list, you may also want to set
 * sfd->ncyls to 0, unless you want to remake a file of the same
 * length.  After a call to delink, you ought to set sfd->cp to NULL to
 * squelch anybody trying to read the now freed cp list.
 */

extern  FILE * lockfopen ();
extern int      quitit ();

delink (head)
	struct dskblk  *head;
{
	struct dskblk  *ptr,
	               *old;
#ifdef INFINITE
	long    loopcnt = 0,
	        toomany = 32768;
#endif INFINITE

	if (head == NULL)
		return;		/* No head */
	for (ptr = head; ptr != NULL && ptr -> flag != EOLIST;) {
		if (ptr != NULL) {
			old = ptr;
			ptr = ptr -> next;
		}
		if (old -> dfn != NULL)
			free (old -> dfn);
		free ((char *)old);
#ifdef INFINITE
		if (loopcnt++ > toomany) {
			fprintf (stderr, "delink: infinite loop\n");
			report ("delink: infinite loop\n");
			exit(1);;
		}
#endif INFINITE
	}
	if (ptr != NULL) {
		if (ptr -> dfn != NULL)
			free (ptr -> dfn);
		free ((char *)ptr);
	}
	return;
}

/*fp points to ALLOCNAME, which is read up and returned.*/

int     free_list;		/* set while touching the free list */

struct dskblk  *
rlist (fp)
	FILE * fp;
{
	extern char *strcpy();
	int     nmatch;
	char    tmpchr[128];
	struct dskblk  *dp,
	               *head;

	free_list = 1;

	if (fp == NULL) {
		free_list = 0;
		return (NULL);
	}
	if ((head = (struct dskblk  *) malloc ((unsigned) 
	    sizeof (struct dskblk))) == NULL)
		malerr("rlist", 1);
	for (
			dp = head, nmatch = 6; 
			nmatch == 6;
			((dp -> next) -> last = dp), (dp = dp -> next)
	    ) {
		char    buf[BUFSIZ];

		if (fgets (buf, BUFSIZ, fp) == NULL) {
			nmatch = EOF;
			break;
		}
#ifdef DEBUG
		fprintf (stderr, "%s\n", buf);
#endif DEBUG
		nmatch = sscanf (buf, "%c%d%d%d%d%s", &dp -> flag, &dp -> base,
				&dp -> len, &dp -> cd, &dp -> seq, tmpchr);
		if (nmatch != 6)
			break;
		if ((dp -> dfn = (char *) malloc((unsigned)
		    strlen (tmpchr) + 1)) == NULL)
			malerr("rlist", 1);
		(void) strcpy (dp -> dfn, tmpchr);
		dp -> next = (struct dskblk *) malloc ((unsigned)
		    sizeof (struct dskblk));
		if (dp -> next == NULL)
			malerr("rlist", 1);
	}
	if (nmatch != EOF) {
		char    buf[128];
		(void) sprintf (buf, 
			"rlist: error scanning free list, nmatch=%d\n",
				nmatch);
		fprintf (stderr, buf);
		report (buf);
		notify ("dgl", buf);
		exit (1);
		return(NULL);	/* for lint's consumption */
	}
	else {
		dp -> next = head;/*  close the circle */
		dp -> flag = EOLIST;
		dp -> base = -1;
		dp -> len = -1;
		dp -> dfn = (char *) NULL;
		head -> last = dp;
		free_list = 0;
		return (head);
	}
}

cklist(name, head)
	char *name;
	struct dskblk *head;
{
	extern struct sfstab *dirinfo ();
	register long   maxcyls, total = 0;
	char   *getsfile ();
	struct sfstab  *sfs;
	struct dskblk *p;

	if ((sfs = dirinfo (name)) == NULL) {
		fprintf (stderr, "cklist: dirinfo failed\n");
		return (-1);
	}
	maxcyls = sfs -> devlen;
	for (p = head; p != NULL && p->flag != EOLIST; p = p->next)
		total += p->len;
	if (total != maxcyls) {
		fprintf(stderr, "cklist: block count error, %d != %d\n",
			total, maxcyls);
		exit(1);
		return(-1);	/* for lint's consumption */
	} else
		return(0);
}

/*
 * freelist - return circular allocation arena to garbage dump 
 */

freelist (listp)
struct dskblk  *listp;
{
	struct dskblk  *dp;
#ifdef INFINITE
	long    loopcnt = 0,
	        toomany = 32768;
#endif INFINITE

	if (listp == NULL)
		return;
	for (dp = listp -> next; dp != NULL; dp = dp -> next) {
		if (dp -> flag == EOLIST)
			break;
		if (dp -> last != NULL) {
			if (dp -> last -> dfn != NULL)
				free (dp -> last -> dfn);
			free ((char *)dp -> last);
		}
#ifdef INFINITE
		if (loopcnt++ > toomany) {
			fprintf (stderr, "freelist: infinite loop\n");
			report ("freelist: infinite loop\n");
			exit(1);
		}
#endif INFINITE
	}
	if (dp != NULL) {
		if (dp -> dfn != NULL)
			free (dp -> dfn);
		free ((char *)dp);
	}
}

/* 
 * wlist - writes the list to disk, frees it,  closes it. 
 * Figures out what file system to write it on from name.
 * If dealloc is true, delete the list, else leave it.
 */

int     interlock;
FILE * lockfp;

wlist (listp, name, dealloc)	/* Write and deallocate list. */
	struct dskblk  *listp;
	char   *name;
	int     dealloc;
{
	extern char   *getsfile (); 
	FILE * fp;
	struct dskblk  *dp;
	char	*allocname;
#ifdef INFINITE
	long    loopcnt = 0,
	        toomany = 32768;
#endif INFINITE

	if (listp == NULL) {
		fprintf (stderr, "wlist: listp == NULL\n");
		return (-1);
	}
	if (interlock) {	/* locked for multi-routine lock */
	/* lockfp must be open for read/write */
		if ((fp = lockfp) == NULL) {
			fprintf (stderr, "wlist: lockfp == NULL\n");
			return (-1);
		}
	/* prepare to write over the file */
		if (fseek (fp, 0L, 0) < 0) {
			perror ("fseek");
			return (-1);
		}
		if (ftruncate (fileno (fp), 0) < 0) {
			perror ("ftruncate");
			return (-1);
		}
	}
	else {
		if (name == NULL)
			return (-1);
		setsfile (name);
		allocname = getsfile (ALLOCNAME);
		if ((fp = lockfopen (allocname, "w")) == NULL) {
			fprintf (stderr, "wlist: lockfopen failed\n");
			return (-1);
		}
	}
	for (dp = listp; dp -> flag != EOLIST; dp = dp -> next) {
		fprintf (fp, "%c\t%d\t%d\t%d\t%d\t%s\n",
				dp -> flag, dp -> base, dp -> len,
				dp -> cd, dp -> seq, dp -> dfn);
#ifdef INFINITE
		if (loopcnt++ > toomany) {
			fprintf (stderr, "wlist: infinite loop\n");
			report ("wlist: infinite loop\n");
			return (-1);
		}
#endif INFINITE

	}
 /* free the list */
	if (dealloc)
		freelist (listp);
	if (lockfclose (fp) != 0) {
		fprintf (stderr, "wlist: lockfclose failed\n");
		return (-1);
	}
	interlock = 0;
	lockfp = NULL;
	return (0);
}


/* 
 * sfgetfl - get free list, figures out where to get it from
 * whatever is in name.
 */

struct dskblk  *
sfgetfl (name, flag)
	char   *name;
	int     flag;
{
	FILE * fp;
	struct dskblk  *makenew (), *rlist (), *head;
	char   *getsfile (), *allocname;
	struct sfstab  *sfs;
	extern struct sfstab  *dirinfo ();

	setsfile (name);
	allocname = getsfile (ALLOCNAME);
	if ((fp = fopen (allocname, "r+")) == NULL) {
	/* no initial free disk file */
	/* make one up */
		if (flag) {
			register long   maxcyls;

			fprintf (stderr, "sfgetfl: new free storage list\n");
			if ((sfs = dirinfo (name)) == NULL) {
				fprintf (stderr, "sfgetfl: dirinfo failed\n");
				return (NULL);
			}
			maxcyls = sfs -> devlen;
			head = makenew (maxcyls);
		}
		else {
			fprintf (stderr, "sfgetfl: nonexistent file: %s\n",
					allocname);
			return (NULL);
		}
	}
	else {
		if (lockfid (fp) < 0) {
			fprintf (stderr, "sgetfl: lockfid failed on %s\n",
					allocname);
			return (NULL);
		}
		if ((head = rlist (fp)) == NULL) {
			fprintf(stderr, "sfgetfl: rlist failed\n");
			exit(1);
		}
		if (cklist(allocname, head) != 0) {
			fprintf(stderr, "sfgetfl: %s: corrupted free list\n",
				allocname);
			exit(1);
		}
		if (interlock != 0)	/* leave lock on the file */
			lockfp = fp;	/* save it for writing */
		else {
			if (lockfclose (fp) < 0) {
				fprintf (stderr,
					"sgetfl: lockfclose failed on %s\n",
					allocname);
			}
		}
		if (head == NULL) {
			fprintf (stderr, "sfgetfl: error reading %s\n",
				allocname);
			return (NULL);
		}
	}
	free (allocname);
	return (head);
}

/* 
 * catfree takes a pointer to a disk block structure, and if it and the next
 * block is UNUSED, and the second butts up against the first block, then it 
 * concatenates the next one with this one, doing all the appropriate magic
 * and deleting the swallowed link.
*/

catfree (dp)
struct dskblk  *dp;
{
	struct dskblk  *nxt = dp -> next;

	if ((dp -> flag == UNUSED) && (nxt -> flag == UNUSED) &&
			((dp -> base + dp -> len) == nxt -> base)) {
		dp -> len = dp -> len + nxt -> len;
		dp -> next = nxt -> next;
		(nxt -> next) -> last = dp;
		free ((char *)nxt);
		return (1);
	}
	return (0);
}


/* zapfree makes successive calls to catfree to zap the entire list */
/* returns # of blocks zapped */

long
zapfree (head)
	struct dskblk  *head;
{
	struct dskblk  *ptr;
	long    n;
#ifdef INFINITE
	long    toomany = 32768;
#endif INFINITE

	if (head == NULL)
		return (-1);
	for (n = 0, ptr = head; ptr -> flag != EOLIST; ptr = ptr -> next) {
		while (catfree (ptr) > 0) {
			n++;
#ifdef INFINITE
			if (n >= toomany) {
				fprintf (stderr, "zapfree: infinite loop\n");
				report ("zapfree: infinite loop\n");
				return (-1);
			}
#endif INFINITE
		}
	}
	return (n);
}


/* 
 * makenode invents a new node on the list, sets its base, length and
 * flag, and links it to next and last
 */

struct dskblk  *
makenode (last, next, base, len, flag, dfn, sfd, cd, seq)
	struct dskblk  *last,
		       *next;
	long    base,
		len;
	int     flag;
	char   *dfn;
	struct sndesc  *sfd;
	long    cd,
		seq;
{
	extern char *strcpy();
	struct dskblk  *ptr;

	if ((ptr = (struct dskblk   *) malloc ((unsigned)
	    sizeof (struct dskblk))) == NULL)
		malerr("makenode", 1);
	ptr -> last = last;
	ptr -> next = next;
	ptr -> base = base;
	ptr -> len = len;
	ptr -> flag = flag;
	ptr -> dsksfd = sfd;
	if (dfn != NULL) {
		if ((ptr -> dfn = (char *) malloc ((unsigned)strlen (dfn) + 1)) 
		    == NULL)
			malerr("makenode", 1);
		(void) strcpy (ptr -> dfn, dfn);
	}
	ptr -> cd = cd;
	ptr -> seq = seq;
 /* link it into its list if possible */
	if (last != NULL)
		last -> next = ptr;
	if (next != NULL)
		next -> last = ptr;
	return (ptr);
}


/* makenew creates a new free list from scratch */

struct dskblk  *
                makenew (maxcyls)
long    maxcyls;
{
	struct dskblk  *ptr = NULL,
	               *eolist = NULL;
	if ((ptr = makenode (ptr, ptr, 0L, maxcyls, UNUSED, 
		"none", NULL, 0, -1))
			== NULL)
		return (NULL);
	if ((eolist = makenode (ptr, ptr, -1L, -1, EOLIST, "none", 
	    NULL, 0L, -1L)) == NULL)
		return (NULL);
	ptr -> next = eolist;	/*  close the circle */
	ptr -> last = eolist;
	return (ptr);
}


/* 
 * Search circular list for either largest free block or
 * one >= cyls.  If cyls == 0, returns pointer to largest block found. 
 * If cyls > 0, returns pointer to block of size cyls, if one can be found,
 * otherwise it returns NULL.  The block is marked ALLOCATED.
 * If cyls < first large enough free space found,
 * a new block is made for remainder, marked UNUSED
 * and placed just after claimed block.  Note: the ptr returned is to
 * the block on the list.  You must copy the node with makenode() if
 * you want to diddle with it, such as putting it on an sfd or the like,
 * or you will screw up the free list.
*/

struct dskblk  *
claim (head, cyls)
	struct dskblk  *head;
	long    cyls;
{				/* search clockwise if direction >= 0,
				   else counter- */
	struct dskblk  *dp;
#ifdef INFINITE
	long    loopcnt = 0,
	        toomany = 32768;
#endif INFINITE

	if (cyls < 0)
		return (NULL);
	if (cyls == 0) {	/* find biggest block */
		struct dskblk  *maxblk = NULL;
		long    maxsiz = 0;

		for (dp = head; dp -> flag != EOLIST; dp = dp -> next) {
			if ((dp -> flag == UNUSED) && (dp -> len > maxsiz)) {
				maxblk = dp;
				maxsiz = dp -> len;
			}
			 (void) catfree (dp);
#ifdef INFINITE
			if (loopcnt++ > toomany) {
				fprintf (stderr, "claim 1: infinite loop\n");
				report ("claim 1: infinite loop\n");
				return (NULL);
			}
#endif INFINITE
		}
		if (maxblk != NULL)
			maxblk -> flag = ALLOCATED;
		return (maxblk);
	}
	else {			/* find free space >= cyls */
		for (dp = head; dp -> flag != EOLIST; dp = dp -> next) {
			if ((dp -> flag == UNUSED) && (dp -> len >= cyls)) {
				if (ckblk(dp) != 0) {
					fprintf(stderr, 
					    "claim: block allocation error\n");
					exit(1);
				}
				if (dp -> len > cyls) {
					if (makenode (dp, dp -> next,
						dp -> base + cyls,
						dp -> len - cyls, UNUSED,
						"none", NULL, 0L, -1L) == NULL)
							return (NULL);
					dp -> len = cyls;
				}
				dp -> flag = ALLOCATED;
				return (dp);
			}
#ifdef INFINITE
			if (loopcnt++ > toomany) {
				fprintf (stderr, "claim 2: infinite loop\n");
				report ("claim 2: infinite loop\n");
				return (NULL);
			}
#endif INFINITE
		}
		return (NULL);
	}
}


/*
 * Claim cyls cylinders.  If sfd->rtflag == RT, they must be contiguous,
 * and allocsf returns ptr to dskblk structure.  Otherwise  allocsf
 * returns NULL.  NULL returned on all failures.  seq is a sequence number
 * for the block claimed, which is only useful for non-contiguous files.
 * For consistency, set it to -1 for realtime files, and for
 * non-contiguous files, start with seq = 0 and count up.  This helps a
 * little when looking at ALLOCNAME to figure out what is what.  The last
 * argument, if not NULL, is taken to be an alternate in-core free list to
 * use instead of reading in the master free list from ALLOCNAME.  Use of
 * an alternate free list inhibits it from writing out the updated list to
 * the disk, assuming you are doing some non-standard usage.  Write the
 * list yourself when ready with wlist();
 */

struct dskblk  *
allocsf (sfd, cyls, seq, althead)
	struct sndesc  *sfd;
	long    cyls,
		seq;
	struct dskblk  *althead;
{
	extern char *strcpy();
	struct dskblk  *head,
	               *ptr;

	if (althead == NULL) {	/* ordinary call */
		ignall ();
		do {
			interlock = 1;
			head = sfgetfl (sfd -> sfn, 0);
			if (zapfree (head) < 0) {
				fprintf (stderr,
					"allocsf: error in free list\n");
				return (NULL);
			}
		/* scribble on the free list */
			if ((ptr = claim (head, cyls)) == NULL)
				return (NULL);
		} while (ptr == NULL);
	}
	else {			/* sombody building new list */
		if ((ptr = claim (althead, cyls)) == NULL)
			return (NULL);
	}


 /* fix up claimed node for free list */
	ptr -> dfn = (char *) malloc ((unsigned)strlen (sfd -> sfn) + 1);
	if (ptr -> dfn == NULL)
		malerr("allocsf", 1);
	(void) strcpy (ptr -> dfn, sfd -> sfn);
	ptr -> dsksfd = sfd;
	ptr -> cd = sfd -> cdate;
	ptr -> seq = seq;

 /* make copy of node to pass back, will become the link in sfd */
	if (althead != NULL)
		ptr = makenode (NULL, NULL, ptr -> base, ptr -> len, 
			ptr -> flag, ptr -> dfn, sfd, 0L, seq);
	else {			/* do not write out alternate head */
		ptr = makenode (NULL, NULL, ptr -> base, ptr -> len, 
			ptr -> flag, NULL, NULL, 0L, seq);
		if (wlist (head, sfd -> sfn, 1) != 0)
			return (NULL);
	}
	if (althead == NULL)
		catchall (quitit);
	return (ptr);
}


/*
 * allolook(head, cyls, sfd)
 * 	struct dskblk *head;
 * 	long cyls;
 * 	struct sndesc *sfd;
 * {
 * 	freelist(head);
 * 	if (grim_reaper(cyls, sfd->sfn) < 0) {
 * 		fprintf(stderr, "allocsf: %s\n", "not enough room on disk");
 * 		return(-1);
 * 	}
 * 	return(0);
 * }
 */
