/* NOEXTRA will not allow playing beyond EOF. This is a feature of the
   4.2 DAC driver that will support adding silence at the tail of a file. */

/* @(#)playmain.c	1.6	5/1/86	IRCAM (from 1.5	(CARL)	5/17/85	18:18:23) */

# include <sys/types.h>
# include <stdio.h>
# include <sys/stat.h>
# include <signal.h>
# include <dsc.h> 
# include <errno.h>
# include <sys/wait.h>
# include <carl/libsf.h>
#include <sfheader.h>
# include "play.h"

/* #define FORKYOU		/* allows recovery from interrupts */
#define IGNORE 1	/* for crack() */
#define DONT_IGNORE 0

extern int errno;
extern int grepeat;	/* from dsdac.c */
extern char shellstr[];	/* from interact.c */

int intr;		/* interactive flag set? */
int verbose, wizard, prompt=0, synchro=0,disklocking=0, locklevel;
int syncval = 30;
char defconv[] = "1,2,3,4";
extern float sfexpr();
int nfiles;

main(argc, argv)
	int argc; char *argv[];
{
	int shutup(), icatch(), qcatch(), ichild(), qchild();
	union wait status;
	int pid, w;


	nfiles = playparse(argc, argv);	/* read down command line */

#ifdef FORKYOU
	if ((pid = fork()) == 0) {
#endif FORKYOU
		if (nfiles == 0) 
			exit(1); 
		else {
			int i;

			for (i = 0; i < nfiles; i++)
				if (setup(i, pf[i].cfile)) 
					{ closeall(); exit(1); }
		}
		if (intr) {
#ifdef FORKYOU
			signal(SIGINT, ichild); 
			signal(SIGQUIT, qchild); 
#endif FORKYOU
			interact();
		} else
			playall();
		closeall();
		exit(0);
#ifdef FORKYOU
	}

	/* wait for process, or interrupts */
	if (intr) {
		signal(SIGINT, icatch); 
		signal(SIGQUIT, qcatch); 
	}
	else {
		signal(SIGINT, shutup); 
		signal(SIGQUIT, shutup); 
	}
	while (((w = wait(&status)) != pid && w != -1) 
		|| (w == -1 && errno == EINTR)) 
			/* empty */; 
	if (w == -1) 
		exit(1);
	exit(0);
#endif FORKYOU
}

shutup() 
{
#ifdef debug
	printf("shutup\n");
#endif debug
	if (dsinit(DEVDSR0) < 0) 
		perror("shutup"); 
}

extern int playing;	/* from dsdac() */

icatch() 
{
#ifdef debug
	printf("icatch\n");
#endif debug
	signal(SIGINT, icatch); 
	if (dsinit(DEVDSR0) < 0) 
		perror("icatch"); 
}

extern int autoseg;

qcatch() 
{
#ifdef debug
	printf("qcatch\n");
#endif debug
	signal(SIGQUIT, qcatch);
}

int inthappened; /* set by ichild(), tells dsdac() to stop */

ichild() 
{
#ifdef debug
	printf("ichild\n");
#endif debug
	signal(SIGINT, ichild); 
	inthappened++; 
	grepeat = 0;
}


qchild() 
{
#ifdef debug
	printf("qchild\n");
#endif debug
	signal(SIGQUIT, qchild);
	if (autoseg) addseg();
}


setup(find, file)
	int find; char *file;
{
	extern char *strcpy();
	int sfd;
	char name[128];			/* sound file name */
	double sampfac;
	SFHEADER hd;

	strcpy(name, file);

	sfd = open(name, 0);	/* open file */
	if (sfd < 0 ) {
		printf(stderr,"play: opensf failed on %s\n",name); 
		return(1);
	}
	if(rheader(sfd,&hd)) {
		printf("Bad header read\n");
		goto bad;
	}
	if(!ismagic(&hd)) {
		printf("%s not a soundfile? (magic = %d)\n",
			name,hd.sfinfo.sf_magic);
		goto bad;
	}

	sfmagic(&pf[find].header) = sfmagic(&hd);
	SFSRATE(pf[find].header) = SFSRATE(hd);
	SFCHANS(pf[find].header) = SFCHANS(hd);
	SFCLASS(pf[find].header) = SFCLASS(hd);
	if(SFCLASS(hd) != SF_SHORT) {
		fprintf(stderr,"%s - sample size not playable (%d bytes/sample)\n",
			name,SFCLASS(hd));
		goto bad;
	}
	if(setsize(find))
		goto bad;;


	/* generate correct begin and end timeing now that we know sr and nc */
	sampfac = SFSRATE(pf[find].header) * SFCHANS(pf[find].header);
	if (pf[find].cbegin != NULL)
		pf[find].begin = sfexpr(pf[find].cbegin, sampfac);
	else
		pf[find].begin = 0;
	if (pf[find].cend != NULL)
		pf[find].end = sfexpr(pf[find].cend, sampfac);
	else
		pf[find].end = pf[find].size;

	pf[find].sfp = sfd;
	if (pf[find].srover == 0.0) 
		pf[find].srover = SFSRATE(pf[find].header);
	if (pf[find].cconvover[0] == NULL)	/* no override set */
		strncpy(pf[find].convover, defconv, SFCHANS(pf[find].header)*2-1);
	else
		strcpy(pf[find].convover,pf[find].cconvover);
	/* copy header */
	return(0);
bad:	close(sfd);
	return(-1);
}


/* !!4.2** */
/* step through linked list of open files */
playall()
{
	int i;

	for (i = 0; i<MAXFILES; i++) {
		if(pf[i].sfp > 0)
			playi(i);
	}
}

playi(i)
	int i;
{
	float timefac;
	double fbeg, fend;
	int sfd;

	sfd = pf[i].sfp;
	setsize(i);
	timefac = SFSRATE(pf[i].header) * SFCHANS(pf[i].header);
	fbeg = pf[i].begin/timefac;
	fend = pf[i].end/timefac;
	/* See if size changed */
	if (!wizard) {
		if (cktimes(i, &fbeg, &fend))
			return(-1);
	}
	pf[i].begin = fbeg * timefac;
	pf[i].end = fend * timefac;
	if (shellstr[0] != '\0') {
		int ret;
		if ((ret = shellcmd(i, pf[i].begin, pf[i].end)) < 0) 
			fprintf(stderr, "shellcmd failed\n");
		return(ret);
	} else
		if (play(i, pf[i].begin, pf[i].end, pf[i].silence, 
			pf[i].repeat, pf[i].prompt, pf[i].srover, 
			pf[i].fltover, pf[i].convover) < 0) {
				fprintf(stderr, "play failed\n");
				return(-1);
		}
	return(i);
}

/* close all open files  4.2 ***** */
closeall()
{
	int i;

	for ( i = 0 ; i < MAXFILES ; i++)
		if(pf[i].sfp > 0)
			close(pf[i].sfp);
}

/*
 * playparse - read play command line.  Syntax is:
 * play -p {[-sN] [-rN] [file]}
 * where -s specifies N secs. silence preceeding the filename which follows
 * on the command line,
 * -r specifies N repetitions (with no silence between) of the file which
 * follows on the command line.  The -p flag causes play to wait for a prompt
 * from user before playing first file.
 * The routine fills three arrays, silences,
 * repeats, and files.  Silences and repeats are filed with durations and
 * iterations, files is filled with indicies into the argv array.
 * It returns the number of files.
 */

playparse(argc, argv)
	int argc; char **argv;
{
	char *index(), flag, *option, *strsave();
	register int i, j;
	int glob=1;

	/* init end times to play whole file, no dac, nc, or filter override */
	for (i = 0; i < MAXFILES; i++) {
		pf[i].end = -1; 	/* play to end of file */
		pf[i].fltover = -1; 	/* use default filters */
		pf[i].bno = -1;		/* start at beginning block */
		pf[i].cnt = -1;		/* play all blocks */
	}

	for (i = 1, j = 0; i < argc; i++) {
		if (*argv[i] == '-'){	/* itsa flag */
			flag = *(argv[i]+1);
			option = argv[i]+2;
			switch (flag) {
				case 'b': 	/* begin */
					pf[j].cbegin = option; break;
				case 'e': 	/* end */
					pf[j].cend = option; break;
				case 'g':	/* make flags local only */
					glob=0; break;
				case 'h': 	/* wants some help */
					helpmessage(); break;
				case 'i':	/* run files interactively */
					intr++; break;
				case 'p': 	/* wants a prompt */
					pf[j].prompt = 1;
					break;
				case 'q': 	/* itsa silence */
					pf[j].silence = sfexpr(option, 1.0); 
					break;
				case 'r': 	/* itsa repeat */
					pf[j].repeat = sfexpr(option, 1.0); 
					break;
				case 's':	/* tape synchronisation */
					synchro = 1; 
					syncval = sfexpr(option, 1.0);
					if(syncval < 0 || syncval > (60*6)) {
						printf("Syncval out of range (%d)\n",
							syncval);
						exit(1);
					}
					if(syncval == 0)
						syncval = 30;
					break;
				case 'v': 	/* verbose */
					verbose++; break;
				case 'B': 
					pf[j].bno = sfexpr(option, 1.0); break;
				case 'C': 
					pf[j].cnt = sfexpr(option, 1.0); break;
				case 'D': 	/* dac override */
					strcpy(pf[j].cconvover, option); break;
				case 'F': 	/* filter override */
					pf[j].fltover = sfexpr(option, 1.0); 
					break;
				case 'R': 	/* sample rate override */
					pf[j].srover = sfexpr(option, 1.0); 
					break;
				case 'L': 	/* Lock disks */
					disklocking = 1;
					locklevel = sfexpr(option, 1.0);
					break;
				default:
					fprintf(stderr, "illegal flag: %s\n", 
						argv[i]);
					helpmessage();
				}
		} else {
			if (j >= MAXFILES) {
				fprintf(stderr,"play: too many files, max=%d\n",
					MAXFILES);
				exit(1);
			}
			pf[j].cfile = strsave(getsfname(argv[i]));
			if(checkfile(pf[j].cfile)) 
				free(pf[j].cfile);
			else { /* good file */
				j++; /* Next slot */
				if (glob) cppsf(j, j-1); /* setup for next file */
			}
		}
	}
	return(j);
}

/*
 * in playparse(), if the -g flag is set, cppsf() gets called to pre-set the
 * next pf[] array entry to the values of those for the previous file.
 */

cppsf(to, from)
	int to, from;
{
	pf[to].silence = pf[from].silence;
	pf[to].repeat = pf[from].repeat;
	pf[to].prompt = pf[from].prompt;
	pf[to].fltover = pf[from].fltover;
	pf[to].cbegin = pf[from].cbegin;
	pf[to].cend = pf[from].cend;
	strcpy(pf[to].cconvover, pf[from].cconvover);
}

helpmessage()
{
printf("%s%s%s%s%s%s%s%s%s%s%s%s%s", 
"usage: play [[flags] [filename]]...\n",
" -g make all flags apply only to immediately following file\n",
" -p prompts you before starting to play file\n",
" -i interactive play mode\n",
" -qN inserts N secs. silence before file\n",
" -rN repeats file N times (no gap between iterations)\n",
" -s tape synchronisation\n",
" -bN start file at time N\n",
" -eN end file at time N\n",
" -RN force file to be N sampling rate\n",
" -FN force use of filter N (0 =20KHz, 1 =10KHz, 2 =bypass, 3 =5KHz)\n",
" -Dw,x,y,z... set dacs to use; e.g., -D2,1 plays reverse channel stereo\n",
" -v  describe in detail what play is doing\n"
);
exit(0);
}


/* 
 * check time in seconds against file descriptior and some
 * other reasonable constraints.
 */

cktimes(file, fbegin, fend)
	float *fbegin, *fend;
{
	register double endtime = pf[file].size / (SFCHANS(pf[file].header) * 
			SFSRATE(pf[file].header));
	register int rtn = 0;
	if (*fend < *fbegin) {
		printf("error: begin must be > end\n"); 
		rtn--; 
	}
	if (*fbegin < 0) 
		*fbegin = 0.0;
#ifdef NOEXTRA
	if (*fend > endtime) 
		*fend = endtime;
#endif
	return(rtn);
}

setsize(fil)
{
	struct stat st;
	long newsize;
	int sfd = pf[fil].sfp;

	/* update file size */
	if(fstat(sfd,&st)) {
		printf("bad stat \n");
		return(-1);
	}
	newsize = SFBSIZE(st)/SFCLASS(pf[fil].header); /* in samples */
	if(pf[fil].end == pf[fil].size) /* Was old end at EOF? */
		pf[fil].end = newsize;
	pf[fil].size = newsize;

#ifdef NOEXTRA
	if(!wizard)
		if(pf[fil].end >  pf[fil].size) 
			pf[fil].end = pf[fil].size;
#endif NOEXTRA

	pf[fil].blksize = st.st_blksize; /* Native conversion size of disk */
	return(0);
}
